<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\ImageInterface;
use Pongho\Core\Kernel;
use Pongho\Utilities\Inflector;

/**
 * Modello per i file.
 *
 * @property int    $id
 * @property int    $site_id
 * @property string $name
 * @property int    $size
 * @property string $mimetype
 * @property string $file
 * @property string $path
 */
class File extends Base implements ImageInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'files';

    /**
     * Callback `before_save`.
     *
     * @var array
     */
    public static $before_save = ['fixPath'];

    /**
     * @var Site
     */
    protected $_site;

    /**
     * {@inheritdoc}
     */
    public function __get($name)
    {
        if ($name === 'site') {
            return $this->getSite();
        }

        return parent::__get($name);
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return int
     */
    public function getSiteId()
    {
        return $this->site_id;
    }

    /**
     * @return Site
     */
    public function getSite()
    {
        if ($this->_site === null) {
            /** @var Manager\SiteManagerInterface $site_manager */
            $site_manager = Kernel::instance()->getContainer()->get('site_manager');

            $this->_site = $site_manager->findById($this->site_id);
        }

        return $this->_site;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getMimeType()
    {
        return $this->mimetype;
    }

    /**
     * @return string
     */
    public function getCompleteFileName()
    {
        return $this->file;
    }

    /**
     * {@inheritdoc}
     *
     * @param boolean $delete_file Indica se eliminare o meno il file associato.
     */
    public function delete($delete_file = true)
    {
        if ($delete_file) {
            $file_path = $this->filepath();

            if (file_exists($file_path)) {
                unlink($file_path);
            }

            $this->clean();
        }

        return parent::delete();
    }

    /**
     * {@inheritdoc}
     */
    public function updateAttributes(array $attributes)
    {
        parent::updateAttributes($attributes);

        if ($attributes === []) {
            return;
        }

        $this->clean();
    }

    /**
     * Cerca un file in base al nome ed al percorso (campi `file` e `path` in tabella).
     *
     * @param string  $name    Nome del file caricato.
     * @param string  $path    Cartella di destinazione del file.
     * @param integer $site_id ID del sito di riferimento.
     * @return File
     */
    public static function findByName($name, $path, $site_id)
    {
        $path = '/' . trim($path, '/');

        return static::first([
            'conditions' => [
                'file = :file AND path = :path AND site_id = :site',
                'file' => $name,
                'path' => $path,
                'site' => $site_id,
            ],
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function path()
    {
        return $this->getSite()->getUploadsUrl($this->getFixedPath() . '/' . $this->file);
    }

    /**
     * {@inheritdoc}
     */
    public function filepath()
    {
        return $this->getSite()->getUploadsPath($this->getFixedPath() . '/' . $this->file);
    }

    /**
     * {@inheritdoc}
     */
    public function size()
    {
        return Inflector::sizize($this->size);
    }

    /**
     * Corregge il percorso del file corrente.
     *
     * Si assicura che il percorso `uploads_path` del sito non sia presente.
     *
     * Questo metodo è lanciato in automatico al momento del salvataggio del modello.
     */
    public function fixPath()
    {
        $this->path = $this->getFixedPath();
    }

    /**
     * Restituisce il percorso del file corrente, senza il percorso `uploads_path` del sito.
     *
     * Questo metodo non modifica il valore dell’attributo `path` del modello.
     *
     * @return string
     */
    public function getFixedPath()
    {
        $uploads_path = $this->getSite()->getUploadsPath();

        if (str_starts_with($this->path, $uploads_path)) {
            trigger_error(
                'Absolute path on "File" model is deprecated, use the relative path instead.',
                E_USER_DEPRECATED
            );

            return substr($this->path, strlen($uploads_path));
        }

        return $this->path;
    }

    /**
     * {@inheritdoc}
     */
    public function addSize($size)
    {
        trigger_error('Method ' . __METHOD__ . ' is deprecated.', E_USER_DEPRECATED);

        return $this->path . '/' . $this->getFileNameWithSize($size);
    }

    /**
     * {@inheritdoc}
     */
    public function getSizePermalink($size = null)
    {
        return $this->getSite()->getImagesUrl($this->path . '/' . $this->getFileNameWithSize($size));
    }

    /**
     * @param string $size
     * @return string
     */
    private function getFileNameWithSize($size = null)
    {
        if ($size) {
            $info = pathinfo($this->file);

            return $info['filename'] . '-' . $size . '.' . $info['extension'];
        }

        return $this->file;
    }

    /**
     * Elimina le immagini in frontend generate da Pongho.
     */
    private function clean()
    {
        /** @var \Pongho\Template\Theme $view */
        $view = Kernel::instance()->getContainer()->get('theme_view');

        foreach ($view->getOption('image_sizes', []) as $size => $opts) {
            $path = $this->getSite()->getImagesPath($this->path . '/' . $this->getFileNameWithSize($size));

            if (file_exists($path)) {
                unlink($path);
            }
        }
    }
}
