<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Mailer;

use Application\Core\Entity\SiteInterface;
use Pongho\Core\LocalizationInterface;
use RuntimeException;
use Symfony\Component\Mime\Address;
use Symfony\Component\Mime\Email;
use Symfony\Component\Mime\Part\DataPart;

/**
 * Classe di aiuto per il Mailer.
 */
final class Helper
{
    /**
     * @var SiteInterface
     */
    protected $site;

    /**
     * @var LocalizationInterface
     */
    protected $localization;

    public function __construct(SiteInterface $site, LocalizationInterface $localization)
    {
        $this->site = $site;
        $this->localization = $localization;
    }

    /**
     * Estrae gli indirizzi email da una stringa che può contenerne vari in diverse varianti, separati da virgola.
     *
     * @return Address[]
     */
    public function extractAddressesFromString(string $string): array
    {
        $addresses = [];
        foreach (array_map('trim', explode(',', $string)) as $token) {
            if ('' !== $token) {
                $addresses[] = Address::create($token);
            }
        }

        return $addresses;
    }

    public function replaceVars(string $text, array $vars = []): string
    {
        if ($text === '') {
            return '';
        }

        // Traduzioni
        $text = preg_replace_callback(['#{L_(.*?)}#', '#%L_(.*?)%#'], function (array $matches) {
            return $this->localization->get($matches[1]);
        }, $text);

        // Opzioni del sito
        $text = preg_replace_callback(['#{C_(.*?)}#', '#%C_(.*?)%#'], function (array $matches) {
            if ('site_name' === $matches[1]) {
                return $this->site->getName();
            }

            return $this->site->getOption($matches[1]);
        }, (string) $text);

        // Variabili
        $vars = array_merge($this->getDefaultVars(), $vars);

        return preg_replace_callback(['#{(.*?)}#', '#%(.*?)%#'], function (array $matches) use ($vars) {
            foreach ([$matches[1], strtoupper($matches[1]), strtolower($matches[1])] as $key) {
                if (isset($vars[$key])) {
                    return $vars[$key];
                }
            }

            return $matches[0];
        }, (string) $text);
    }

    /**
     * Esegue alcuni controlli sulla email.
     *
     * - Sostituisce la stringa `{COMPANY_LOGO}` con l’embed del logo aziendale.
     */
    public function prepareEmail(Email $email): Email
    {
        $html = $email->getHtmlBody();

        if (is_string($html) && str_contains($html, '{COMPANY_LOGO}')) {
            $logoPath = $this->site->getThemesPath('images/logo-email.png');

            if (!file_exists($logoPath)) {
                throw new RuntimeException(sprintf('Email logo not found on path "%s"', $logoPath));
            }

            $email->addPart(
                (new DataPart(fopen($logoPath, 'r'), 'logo', 'image/png'))->asInline()
            );

            $html = str_replace('{COMPANY_LOGO}', 'cid:logo', $html);
            $email->html($html);
        }

        return $email;
    }

    /**
     * Restituisce le variabili di default da usare nelle email.
     */
    private function getDefaultVars(): array
    {
        // Autore del sito
        if ($this->site->getOption('author_website')) {
            $author = '<a href="' . $this->site->getOption('author_website') . '" style="' . $this->site->getOption('email_link_style') . '">' . $this->site->getOption('author_name') . '</a>';
        } else {
            $author = $this->site->getOption('author_name');
        }

        $date = date('d/m/Y');
        $time = date('H:i');

        return [
            'COMPANY_SITE'        => absolute_url(''),
            'EMAIL_PRIVACY'       => sprintf($this->localization->get('email_privacy'), $this->site->getOption('company_name')),
            'EMAIL_COPYRIGHT'     => sprintf($this->localization->get('email_copyright'), $author),
            'EMAIL_DATE_SEND'     => sprintf($this->localization->get('email_date_send'), $date),
            'EMAIL_DATETIME_SEND' => sprintf($this->localization->get('email_datetime_send'), $date, $time),
        ];
    }
}
