<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\LinkType;

use Application\Core\Entity\LinkTypeInterface;
use Application\Core\Model\Manager\LinkTypeManagerInterface;
use Pongho\Template\Html;

class LinkFacade
{
    /**
     * @var LinkHandlerCollection
     */
    protected $handlers;

    /**
     * @var LinkTypeManagerInterface
     */
    protected $manager;

    /**
     * @var Html
     */
    protected $html;

    public function __construct(LinkHandlerCollection $handlers, LinkTypeManagerInterface $manager, Html $html)
    {
        $this->handlers = $handlers;
        $this->manager = $manager;
        $this->html = $html;
    }

    /**
     * @param int|null $id
     * @param int|null $language_id
     * @return LinkTypeInterface|null
     */
    public function save(array $post, $id = null, $language_id = null)
    {
        $post['language_id'] = $language_id;

        $link_handler = $post['link_handler'];
        $handler = $this->handlers->get($link_handler);

        $link_type = $id ? $this->manager->findById($id) : null;

        if ($link_type) {
            if ($handler->getName() === 'nolink') {
                // Suppongo che la relazione abbia il CASCADE impostato, altrimenti da errore
                $this->manager->delete($link_type);

                return null;
            }
        } elseif ($handler->getName() === 'nolink') {
            return null;
        } else {
            // In ottica DDD la classe LinkType dovrebbe essere un POPO, ma purtroppo invece è un modello
            // di ActiveRecord e l'istanza non è testabile. Quindi preferisco farla creare dal manager.
            $link_type = $this->manager->newModel();
        }

        $link_type->change($link_handler, $handler->getUrl($post), $handler->getDetails($post));

        $this->manager->save($link_type);

        return $link_type;
    }

    /**
     * @param string   $post_name
     * @param int|null $link_type_id
     * @return string
     */
    public function render($post_name, $link_type_id = null)
    {
        $main_select = [];
        $secondary_options = [];

        $handler_render = '';

        $identifier = null;
        if ($link_type = $this->manager->findById($link_type_id)) {
            $identifier = $link_type->getIdentifier();
        }

        /** @var LinkHandlerInterface $handler */
        foreach ($this->handlers as $handler) {
            if ($handler->getGroup()) {
                $main_select[$handler->getGroup()]['optgroup'][$handler->getName()] = $handler->getLabel();
            } else {
                $main_select[$handler->getName()] = $handler->getLabel();
            }

            $secondary_options[$handler->getName()] = $handler->render($post_name, null);

            $handler_render .= sprintf(
                '<div class="link-type link-type-%s" data-name="%s" data-value="%s"></div>',
                $handler->getName(),
                $this->getFieldName($post_name, $handler),
                $link_type ? $handler->getValue($link_type) : null
            );
        }

        $code = '<select class="link-type-select" name="' . $post_name . '[link_handler]">' . $this->html->select($main_select, $identifier) . '</select>';

        return sprintf(
            '<div class="link-type-wrapper" data-language="%s" data-collection="%s"><div>%s</div>%s</div>',
            $this->handlers->getLanguageId(),
            $this->handlers->getName(),
            $code,
            $handler_render
        );
    }

    /**
     * @return string
     */
    public function renderJson()
    {
        $json_options = [];

        /** @var LinkHandlerInterface $handler */
        foreach ($this->handlers as $handler) {
            $json_options[$handler->getName()] = $handler->render('');
        }

        return json_encode($json_options);
    }

    /**
     * @param string               $name
     * @return string
     */
    protected function getFieldName($name, LinkHandlerInterface $handler)
    {
        return $name ? $name . '[' . $handler->getName() . ']' : $handler->getName();
    }
}
