<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Installer;

use Application\Core\Model\Group;
use Application\Core\Model\LanguageModule;
use Application\Core\Model\Module;
use Application\Core\Model\ModuleSite;
use Application\Core\Model\Permit;
use Application\Core\Model\Role;
use Application\Core\Model\Setting;
use Doctrine\DBAL\Exception as DBALException;
use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Schema\Comparator;
use Pongho\DependencyInjection\Container;

/**
 * Class Installer
 *
 * Base per gli installer delle applicazioni
 *
 * @deprecated Utilizzare gli helper.
 * @todo: Completare gli helper per poter rimuovere questa classe.
 */
class Installer
{
    /**
     * @var Container
     */
    protected $container;

    /**
     * @var \Doctrine\DBAL\Schema\Schema
     */
    protected $schema;

    /**
     * @var \Doctrine\DBAL\Schema\Schema
     */
    protected $old_schema;

    public function __construct(Container $container)
    {
        $this->container = $container;
    }

    /**
     * @return $this
     */
    public static function newInstance(Container $container)
    {
        return new static($container);
    }

    /**
     * @return \Doctrine\DBAL\Schema\Schema
     */
    protected function getSchema()
    {
        $this->schema = $this->container->get('connection_schema');
        $this->old_schema = clone $this->schema;

        return $this->schema;
    }

    /**
     * Aggiorna lo schema
     */
    protected function updateSchema()
    {
        /** @var \Doctrine\DBAL\Connection $conn */
        $conn = $this->container->get('connection');

        $diff = $conn->createSchemaManager()->createComparator()->compareSchemas($this->old_schema, $this->schema);
        $platform = $conn->getDatabasePlatform();

        foreach ($platform->getAlterSchemaSQL($diff) as $query) {
            $conn->executeStatement($query);
        }
    }

    /**
     * Crea un permesso se non esiste e restituisce il modello, o restituisce quello esistente
     *
     * @param string $key
     * @param bool   $get_only
     * @return \ActiveRecord\Base|\Application\Core\Model\Permit
     */
    protected function getOrCreatePermit($key, $get_only = false)
    {
        $permit = Permit::findByKey($key);

        if (!$permit && !$get_only) {
            $permit = Permit::create([
                'key'        => $key,
                'is_enabled' => true,
            ]);
        }

        return $permit;
    }

    /**
     * Crea un ruolo se non esiste e restituisce il modello, o restituisce quello esistente
     *
     * @param string $name
     * @param bool   $get_only
     * @param string $description
     * @return \ActiveRecord\Base|Role
     */
    protected function getOrCreateRole($name, $get_only = false, $description = '')
    {
        $role = Role::first(['conditions' => ['name = ?', $name]]);

        if (!$role && !$get_only) {
            $role = Role::create([
                'name'        => $name,
                'description' => $description,
            ]);
        }

        return $role;
    }

    /**
     * @param        $name
     * @param bool   $get_only
     * @param string $description
     * @return \ActiveRecord\Base
     */
    protected function getOrCreateGroup($name, $get_only = false, $description = '')
    {
        $group = Group::first(['conditions' => ['name = ?', $name]]);

        if (!$group && !$get_only) {
            $group = Group::create([
                'name'        => $name,
                'description' => $description,
            ]);
        }

        return $group;
    }

    /**
     * Crea un modulo se non esiste e restituisce il modello, o restituisce quello esistente
     *
     * @param string $controller
     * @param bool   $get_only
     * @return \Application\Core\Model\Module
     */
    protected function getOrCreateModule($controller, $get_only = false)
    {
        $options = [
            'conditions' => [
                'controller = ?',
                $controller,
            ],
            'cachable'   => true,
        ];

        $module = Module::first($options);

        /** @var \Application\Core\Model\Module $module */
        if (!$module && !$get_only) {
            $module = Module::create(['controller' => $controller]);
        }

        return $module;
    }

    /**
     * Elimina un modulo
     *
     * @param string $controller
     */
    protected function deleteModule($controller)
    {
        $module = $this->getOrCreateModule($controller, true);

        if ($module) {
            $options = ['conditions' => ['module_id = ?', $module->id]];

            foreach (LanguageModule::all($options) as $language_module) {
                $language_module->delete();
            }

            foreach (ModuleSite::all($options) as $module_site) {
                $module_site->delete();
            }

            $module->delete();
        }
    }

    /**
     * Crea un modulo per il sito se non esiste e restituisce il modello, o restituisce quello esistente
     *
     * @param int                                       $module_id
     * @param bool                                      $get_only
     * @param string|\Application\Core\Model\ModuleSite $model_class
     * @return \Application\Core\Model\ModuleSite
     */
    protected function getOrCreateModuleSite($module_id, $get_only = false, $model_class = \Application\Core\Model\ModuleSite::class)
    {
        $options = [
            'conditions' => [
                'module_id = :module AND site_id = :site',
                'module' => $module_id,
                'site'   => $this->container->get('site')->id,
            ],
        ];

        $module_site = $model_class::first($options);

        if (!$module_site && !$get_only) {
            $module_site = $model_class::create([
                'module_id'  => $module_id,
                'site_id'    => $this->container->get('site')->id,
                'is_enabled' => true,
            ]);
        }

        return $module_site;
    }

    /**
     * Crea un modulo per il sito se non esiste e restituisce il modello, o restituisce quello esistente
     *
     * @param int    $module_id
     * @param string $path
     * @param string $name
     * @param bool   $get_only
     * @return \Application\Core\Model\LanguageModule
     */
    protected function getOrCreateLanguageModule($module_id, $path, $name, $get_only = false)
    {
        /** @var \Application\Core\Model\LanguageSite $language_site */
        $language_site = $this->container->get('language_site');

        $language_module = LanguageModule::first([
            'conditions' => [
                'language_site_id = :language_site AND module_id = :module',
                'language_site' => $language_site->id,
                'module'        => $module_id,
            ],
        ]);

        if (!$language_module && !$get_only) {
            // Inserisce un nuovo LanguageModule all'ultima posizione, prima delle pagine

            /** @var \Application\Core\Model\LanguageModule $last_module */
            $last_module = LanguageModule::first([
                'conditions' => [
                    'position < :pages AND language_site_id = :language',
                    'pages'    => 99999,
                    'language' => $language_site->id,
                ],
                'order'      => 'position DESC',
                'limit'      => 1,
            ]);

            $position = ($last_module === null) ? 10 : ($last_module->position + 10);

            $language_module = LanguageModule::create([
                'language_site_id' => $language_site->id,
                'module_id'        => $module_id,
                'path'             => $path,
                'name'             => $name,
                'position'         => $position,
                'is_enabled'       => true,
            ]);
        }

        return $language_module;
    }

    /**
     * Da usare in fase di installazione. Registra gli update che altrimenti darebbero un avvertimento, quando invece sono
     * stati già effettuati dall'installer (che deve essere mantenuto al pari degli update)
     */
    protected function fixRegisteredUpdates()
    {
        // Registro le versioni installate
        /** @var Setting $installed_updates */
        $installed_updates = Setting::find('installed_updates');
        $installed = json_decode($installed_updates->value);

        $result = array_unique(array_merge($installed, $this->getAvailableUpdates()));
        sort($result);

        $installed_updates->value = json_encode($result);
        $installed_updates->save();

        // Aggiorno il valore massimo della versione database
        // ATTENZIONE: il database deve essere già stato aggiornato prima di installare l'applicazione, altrimenti
        // blocca l'aggiornamento di altre applicazioni precedentemente installate, e sarà necessario forzarne gli aggiornamenti
        $max = max($result);

        /** @var Setting $last_update */
        $last_update = Setting::find('db_version');
        if ($last_update->value < $max) {
            $last_update->value = $max;
            $last_update->save();
        }
    }

    /**
     * @return array
     */
    protected function getAvailableUpdates($application = null)
    {
        if ($application === null) {
            $namespace = explode('\\', static::class);
            $application = $namespace[1];
        }

        $updates = [];
        $basepath = __DIR__ . '/../../';
        $path = $basepath . $application . '/Updates/*.php';

        foreach (glob($path, GLOB_BRACE) as $file_path) {
            $class_name = 'Application\\' . str_replace('/', '\\', substr($file_path, strlen($basepath), -4));

            $ref = new \ReflectionClass($class_name);

            if ($ref->implementsInterface(\Updates\UpdateInterface::class)) {
                $offset = intval(substr($ref->getShortName(), 6));
                $updates[] = $offset;
            }
        }

        return $updates;
    }

    /**
     * Popola la tabella delle traduzioni.
     *
     * @param string $path
     */
    protected function installTranslationsFromFile($path)
    {
        $connection = $this->container->get('connection');

        $field_key_name = $connection->getDatabasePlatform()->quoteIdentifier('key');

        foreach (json_decode(file_get_contents($path), true) as $translation) {
            $translation[$field_key_name] = $translation['key'];
            unset($translation['key']);

            try {
                $connection->insert(
                    'pongho_translations',
                    $translation,
                    [
                        'language_id'   => ParameterType::INTEGER,
                        'application'   => ParameterType::STRING,
                        'value'         => ParameterType::STRING,
                        $field_key_name => ParameterType::STRING,
                    ]
                );
            } catch (DBALException) {
                // La traduzione esiste già, ignoro l'errore e proseguo

                continue;
            }
        }
    }
}
