<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Installer\Helper;

use Application\Core\Model\Role;
use Doctrine\DBAL\ArrayParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception as DBALException;
use Doctrine\DBAL\ParameterType;

class PermissionsHelper
{
    /**
     * @var Connection
     */
    protected $connection;

    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * @param string $name
     * @param bool   $is_enabled
     */
    public function addPermission($name, $is_enabled = true)
    {
        $this->addPermissions([$name], $is_enabled);
    }

    /**
     * @param bool  $is_enabled
     * @return array
     */
    public function addPermissions(array $keys, $is_enabled = true)
    {
        $key_field_name = $this->connection->quoteIdentifier('key');

        $result = $this->connection->executeQuery(
            "SELECT * FROM pongho_permits WHERE $key_field_name IN (?)",
            [$keys],
            [ArrayParameterType::STRING]
        );

        $permissions = [];
        foreach ($result->fetchAllAssociative() as $row) {
            $permissions[$row['key']] = $row;
        }

        foreach ($keys as $key) {
            if (!isset($permissions[$key])) {
                $this->connection->insert(
                    'pongho_permits',
                    [
                        $key_field_name => $key,
                        'is_enabled'    => $is_enabled,
                    ],
                    [
                        $key_field_name => ParameterType::STRING,
                        'is_enabled'    => ParameterType::BOOLEAN,
                    ]
                );
            } elseif ($permissions[$key]['is_enabled'] !== $is_enabled) {
                $this->connection->update(
                    'pongho_permits',
                    [
                        'is_enabled' => $is_enabled,
                    ],
                    [
                        'id' => $permissions[$key]['id'],
                    ],
                    [
                        'id'         => ParameterType::INTEGER,
                        'is_enabled' => ParameterType::BOOLEAN,
                    ]
                );
            }
        }

        $result = [];
        foreach ($permissions as $key => $permission) {
            $result[$key] = (int) $permission['id'];
        }

        return $result;
    }

    /**
     * @param string $oldKey
     * @param string $newKey
     */
    public function renamePermission($oldKey, $newKey)
    {
        $key_field_name = $this->connection->quoteIdentifier('key');

        $newId = $this->findPermitId($newKey);
        $oldId = $this->findPermitId($oldKey);

        if ($newId && $oldId) {
            // Seleziono i ruoli interessati dai due permessi e li riassocio solo al nuovo.
            $result = $this->connection->executeQuery(
                'SELECT role_id FROM pongho_permits_roles WHERE permit_id IN (:new, :old) GROUP BY role_id',
                [
                    'new' => $newId,
                    'old' => $oldId,
                ]
            );

            $roles = [];
            foreach ($result->fetchAllAssociative() as $role) {
                $roles[] = $role;
            }

            if ($roles) {
                $rolesStr = implode(', ', $roles);
                $this->connection->executeStatement(
                    "DELETE FROM pongho_permits_roles WHERE permit_id IN ({$newId}, {$oldId}) AND role_id IN ({$rolesStr})"
                );

                foreach ($roles as $role) {
                    $this->connection->insert(
                        'pongho_permits_roles',
                        [
                            'permit_id' => $newId,
                            'role_id'   => $role,
                        ],
                        [
                            'permit_id' => ParameterType::INTEGER,
                            'role_id'   => ParameterType::INTEGER,
                        ]
                    );
                }
            }
        } else {
            $this->connection->update(
                'pongho_permits',
                [
                    $key_field_name => $newKey,
                ],
                [
                    $key_field_name => $oldKey,
                ],
                [
                    $key_field_name => ParameterType::STRING,
                ]
            );
        }
    }

    public function renamePermissions(array $keys)
    {
        foreach ($keys as $oldKey => $newKey) {
            $this->renamePermission($oldKey, $newKey);
        }
    }

    /**
     * @param string $key
     */
    public function removePermission($key)
    {
        $id = $this->findPermitId($key);

        if (!$id) {
            return;
        }

        $this->connection->delete('pongho_permits_roles', ['permit_id' => $id]);
        $this->connection->delete('pongho_permits', ['id' => $id]);
    }

    public function removePermissions(array $keys)
    {
        foreach ($keys as $key) {
            $this->removePermission($key);
        }
    }

    /**
     * @param string $name
     * @param string $description
     * @return int
     */
    public function createRole($name, $description = '')
    {
        $this->connection->insert(
            'pongho_roles',
            [
                'name'        => $name,
                'description' => $description,
            ],
            [
                'name'        => ParameterType::STRING,
                'description' => ParameterType::STRING,
            ]
        );

        return dbal_last_insert_id($this->connection, 'pongho_roles_id_seq');
    }

    /**
     * @param int      $roleId
     * @param string[] $permissions
     */
    public function grants($roleId, array $permissions)
    {
        $roleId = (int) $roleId;
        foreach ($this->addPermissions($permissions) as $permissionId) {
            try {
                $this->connection->insert(
                    'pongho_permits_roles',
                    [
                        'permit_id' => $permissionId,
                        'role_id'   => $roleId,
                    ],
                    [
                        'permit_id' => ParameterType::INTEGER,
                        'role_id'   => ParameterType::INTEGER,
                    ]
                );
            } catch (DBALException) {
                // Il collegamento esisteva già
            }
        }
    }

    /**
     * @param int $roleId
     */
    public function removeRole($roleId)
    {
        $roleId = (int) $roleId;

        $this->connection->update(
            'pongho_users',
            [
                'role_id' => Role::USER_LOGGED,
            ],
            [
                'role_id' => $roleId,
            ]
        );

        $this->connection->delete(
            'pongho_permits_roles',
            [
                'role_id' => $roleId,
            ]
        );

        $this->connection->delete(
            'pongho_roles',
            [
                'id' => $roleId,
            ]
        );
    }

    /**
     * @param string $key
     * @return int
     */
    private function findPermitId($key)
    {
        $key_field_name = $this->connection->quoteIdentifier('key');

        return (int) $this->connection->fetchOne(
            'SELECT id FROM pongho_permits WHERE ' . $key_field_name . ' = :key',
            [
                'key' => $key,
            ]
        );
    }
}
