<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Installer\Helper;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Schema\Comparator;
use Doctrine\DBAL\Schema\Schema;
use Doctrine\DBAL\Schema\Table;

class DatabaseSchemaHelper
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @var Schema
     */
    protected $schema;

    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * @return Schema
     */
    public function get()
    {
        if ($this->schema === null) {
            $this->schema = $this->connection->createSchemaManager()->introspectSchema();
        }

        return $this->schema;
    }

    /**
     * @return bool
     * @throws \Doctrine\DBAL\Exception
     */
    public function update(callable $callable)
    {
        $schema = $this->get();
        $new_schema = clone $schema;

        call_user_func($callable, $new_schema, $this);

        $diff = $this->connection->createSchemaManager()->createComparator()->compareSchemas($schema, $new_schema);
        $platform = $this->connection->getDatabasePlatform();

        foreach ($platform->getAlterSchemaSQL($diff) as $query) {
            $this->connection->executeStatement($query);
        }

        return true;
    }

    /**
     * @param string $columnName
     * @param int    $length
     * @return $this
     */
    public function addStringColumn(Table $table, $columnName, $length = 255)
    {
        $table->addColumn($columnName, 'string', ['length' => $length, 'default' => '']);

        return $this;
    }

    /**
     * @param string $columnName
     * @return $this
     */
    public function addTextColumn(Table $table, $columnName)
    {
        $table->addColumn($columnName, 'text', ['length' => 65535, 'default' => '']);

        return $this;
    }

    /**
     * @param string $columnName
     * @return $this
     */
    public function addJsonColumn(Table $table, $columnName)
    {
        // @todo: Convertire a 'json' quando potremmo aggiornare Doctrine DBAL
        //        Non posso utilizzare neanche 'json_array' perché viene restituito un
        //        errore quando elaboro lo schema ad esempio negli updates.
        $table->addColumn($columnName, 'text', ['default' => '']);

        return $this;
    }

    /**
     * @param string $columnName
     * @param bool   $nullable
     * @return $this
     */
    public function addIntegerColumn(Table $table, $columnName, $nullable = false)
    {
        $table->addColumn($columnName, 'integer', [
            'unsigned' => true,
            'notnull'  => !$nullable,
            'default'  => $nullable ? null : 0,
        ]);

        return $this;
    }

    /**
     * @param string $columnName
     * @param bool   $default
     * @return $this
     */
    public function addBooleanColumn(Table $table, $columnName, $default = false)
    {
        $table->addColumn($columnName, 'boolean', ['default' => $default]);

        return $this;
    }

    /**
     * @param string $columnName
     * @return $this
     */
    public function addPriceColumn(Table $table, $columnName)
    {
        $table->addColumn($columnName, 'decimal', [
            'unsigned'  => true,
            'precision' => 12,
            'scale'     => 4,
            'default'   => 0,
        ]);

        return $this;
    }

    /**
     * @param string $columnName
     * @param bool   $nullable
     * @return $this
     */
    public function addDateTimeColumn(Table $table, $columnName, $nullable = false)
    {
        $table->addColumn($columnName, 'datetime', ['notnull' => !$nullable]);

        return $this;
    }

    /**
     * @param string $columnName
     * @return $this
     */
    public function addPrimaryKey(Table $table, $columnName = 'id')
    {
        $table->addColumn($columnName, 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $table->setPrimaryKey([$columnName]);

        return $this;
    }
}
