<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package Application\Core
 */

namespace Application\Core\I18n\Translation;

use Symfony\Component\Finder\Finder;
use Symfony\Component\Translation\Extractor\ExtractorInterface;
use Symfony\Component\Translation\MessageCatalogue;

class PhpExtractor implements ExtractorInterface
{
    public const MESSAGE_TOKEN = 300;

    public const IGNORE_TOKEN = 400;

    /**
     * Prefix for new found message.
     *
     * @var string
     */
    private $prefix = '';

    /**
     * The sequence that captures translation messages.
     *
     * @var array
     */
    protected $sequences = [
        // Mediante funzione [deprecato]
        [
            '__trans',
            '(',
            self::MESSAGE_TOKEN,
            ')',
        ],
        // Servizio su variabile per il template o utilizzi multipli
        [
            '$translator',
            '->',
            'trans',
            '(',
            self::MESSAGE_TOKEN,
            ')',
        ],
        // Come sopra ma richiesto ad un oggetto (es $this->translator)
        [
            'translator',
            '->',
            'trans',
            '(',
            self::MESSAGE_TOKEN,
            ')',
        ],
        // Servizio richiesto al momento per uso singolo
        [
            'getService',
            '(',
            "'translator'",
            ')',
            '->',
            'trans',
            '(',
            self::MESSAGE_TOKEN,
            ')',
        ],
        // Servizio richiesto al momento per uso singolo
        [
            'get',
            '(',
            "'translator'",
            ')',
            '->',
            'trans',
            '(',
            self::MESSAGE_TOKEN,
            ')',
        ],
        // Servizio richiesto con un metodo per uso singolo
        [
            'getTranslator',
            '(',
            ')',
            '->',
            'trans',
            '(',
            self::MESSAGE_TOKEN,
            ')',
        ],
    ];

    /**
     * {@inheritdoc}
     */
    public function extract($directory, MessageCatalogue $catalog)
    {
        // load any existing translation files
        $finder = new Finder();
        $files = $finder->files()->name('*.php')->in($directory);

        foreach ($files as $file) {
            $this->parseTokens(token_get_all(file_get_contents($file)), $catalog);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function setPrefix($prefix)
    {
        $this->prefix = $prefix;
    }

    /**
     * Normalizes a token.
     *
     * @param mixed $token
     * @return string
     */
    protected function normalizeToken($token)
    {
        if (is_array($token)) {
            return $token[1];
        }

        return $token;
    }

    /**
     * Extracts trans message from PHP tokens.
     *
     * @param array $tokens
     */
    protected function parseTokens($tokens, MessageCatalogue $catalog)
    {
        $tokens_length = count($tokens);
        for ($token_key = 0; $token_key < $tokens_length; $token_key++) {
            foreach ($this->sequences as $sequence) {
                $message = '';

                $skip = 0;
                foreach ($sequence as $sequence_item_key => $sequence_item) {
                    if (!isset($tokens[$token_key + $sequence_item_key + $skip])) {
                        continue;
                    }

                    $value = $this->normalizeToken($tokens[$token_key + $sequence_item_key + $skip]);

                    if (trim(str_replace(["\n", "\r"], '', $value)) === '') {
                        $skip++;

                        if (!isset($tokens[$token_key + $sequence_item_key + $skip])) {
                            continue;
                        }

                        $value = $this->normalizeToken($tokens[$token_key + $sequence_item_key + $skip]);
                    }

                    if ($value == $sequence_item) {
                        continue;
                    } elseif (self::MESSAGE_TOKEN == $sequence_item) {
                        $message = $value;
                    } elseif (self::IGNORE_TOKEN == $sequence_item) {
                        continue;
                    } else {
                        break;
                    }
                }

                $message = trim($message, '\'"');

                if ($message) {
                    $catalog->set($message, $this->prefix . $message);
                    $token_key += count($sequence);
                    break;
                }
            }
        }
    }
}
