<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\I18n;

class LocaleRegistry
{
    /**
     * @var callable[]
     */
    private $factories = [];

    /**
     * @var LocaleInterface[]
     */
    private $locales = [];

    /**
     * @var callable
     */
    private $default;

    /**
     * @param string   $localeName
     * @return $this
     */
    public function register($localeName, callable $factory)
    {
        $this->factories[$localeName] = $factory;

        return $this;
    }

    /**
     * @return $this
     */
    public function registerDefault(callable $factory)
    {
        $this->default = $factory;

        return $this;
    }

    /**
     * @param string $localeName
     * @return LocaleInterface
     * @throws \RuntimeException If no locale exists.
     */
    public function getLocale($localeName)
    {
        if (!isset($this->locales[$localeName])) {
            if (isset($this->factories[$localeName])) {
                $this->locales[$localeName] = $this->validateLocale(call_user_func($this->factories[$localeName]), $localeName);
            } elseif (class_exists(__NAMESPACE__ . '\\Locale\\' . $localeName . '\\Locale')) {
                $class = __NAMESPACE__ . '\\Locale\\' . $localeName . '\\Locale';
                $this->locales[$localeName] = $this->validateLocale(new $class(), $localeName);
            } elseif ($this->default) {
                $this->locales[$localeName] = $this->validateLocale(call_user_func($this->default, $localeName), $localeName);
            } else {
                throw new \RuntimeException(sprintf('No Locale registered for "%s".', $localeName));
            }

            return $this->locales[$localeName];
        }

        return $this->locales[$localeName];
    }

    /**
     * @param LocaleInterface $locale
     * @param string          $localeName
     * @return LocaleInterface
     */
    private function validateLocale($locale, $localeName)
    {
        if (!$locale instanceof LocaleInterface) {
            throw new \RuntimeException(
                sprintf(
                    'The factory for the locale "%s" must be return an implementation of %s',
                    $localeName,
                    LocaleInterface::class
                )
            );
        }

        return $locale;
    }
}
