<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Export\DataExporter;

use Application\Core\Export\DataExporterInterface;
use Application\Core\I18n\Translation\Translator;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\ParameterType;
use Pongho\Utilities\DateTime;

class UsersExporter implements DataExporterInterface
{
    /**
     * @var Translator
     */
    protected $translator;

    /**
     * @var Connection
     */
    protected $connection;

    public function __construct(Translator $translator, Connection $connection)
    {
        $this->translator = $translator;
        $this->connection = $connection;
    }

    /**
     * {@inheritdoc}
     */
    public function headers()
    {
        return [
            $this->translator->trans('Id'),
            $this->translator->trans('Username'),
            $this->translator->trans('Name'),
            $this->translator->trans('Surname'),
            $this->translator->trans('Email'),
            $this->translator->trans('Company'),
            $this->translator->trans('Address'),
            $this->translator->trans('City'),
            $this->translator->trans('Postcode'),
            $this->translator->trans('Province'),
            $this->translator->trans('Telephone'),
            $this->translator->trans('Fax'),
            $this->translator->trans('Website'),
            $this->translator->trans('Role'),
            $this->translator->trans('User status'),
            $this->translator->trans('Subscription date'),
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function export()
    {
        foreach ($this->getStatement()->executeQuery()->fetchAllAssociative() as $row) {
            yield $this->parseRow($row);
        }
    }

    /**
     * @return \Doctrine\DBAL\Statement
     */
    protected function getStatement()
    {
        $sth = $this->connection->prepare($this->getQuery());

        // TODO: move in the query builder
        $sth->bindValue('founder', false, ParameterType::BOOLEAN);

        return $sth;
    }

    /**
     * @return \Doctrine\DBAL\Query\QueryBuilder
     */
    protected function getQuery()
    {
        $qb = $this->connection->createQueryBuilder();

        $qb
            ->select([
                'u.id',
                'u.username',
                'u.name',
                'u.surname',
                'u.email',
                'u.company',
                'u.address',
                'u.city',
                'u.postcode',
                'p.name AS province',
                'u.telephone',
                'u.fax',
                'u.website',
                'r.name AS role',
                'u.is_active',
                'u.created_at',
            ])
            ->from('pongho_users', 'u')
            ->innerJoin('u', 'pongho_roles', 'r', 'r.id = u.role_id')
            ->leftJoin('u', 'pongho_provinces', 'p', 'p.id = u.province_id')
            ->where('u.id <> 1')
            ->andWhere('u.is_founder = :founder')
            ->orderBy('u.created_at')
        ;

        return $qb;
    }

    /**
     * @return mixed
     */
    protected function parseRow(array $row)
    {
        if ($row['is_active'] === null) {
            $row['is_active'] = $this->translator->trans('Pending activation');
        } elseif ((bool) $row['is_active']) {
            $row['is_active'] = $this->translator->trans('Active');
        } else {
            $row['is_active'] = $this->translator->trans('Not active');
        }

        $row['created_at'] = $this->parseDate($row['created_at']);

        return $row;
    }

    /**
     * @param string $date
     * @return string
     */
    protected function parseDate($date)
    {
        if (!$date) {
            return '';
        }

        try {
            return (new DateTime($date))->localeFormat($this->getLocaleDateFormat());
        } catch (\Exception) {
            return '';
        }
    }

    /**
     * @return string
     */
    protected function getLocaleDateFormat()
    {
        return '%x';
    }
}
