<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Entity;

use Application\Core\Model\Role;
use Pongho\Utilities\DateTime;

/**
 * Interface AccountInterface
 */
interface AccountInterface
{
    /**
     * @return int
     */
    public function getId();

    /**
     * @return LanguageInterface
     */
    public function getLanguage();

    /**
     * @return string
     */
    public function getUsername();

    /**
     * @throw \InvalidArgumentException
     * @param string $username
     */
    public function setUsername($username);

    /**
     * @return string
     */
    public function getEmail();

    /**
     * @throw \InvalidArgumentException
     * @param string $email
     */
    public function setEmail($email);

    /**
     * Restituisce la versione criptata della password
     *
     * @return string
     */
    public function getPassword();

    /**
     * @throw \InvalidArgumentException
     * @param string $password
     */
    public function setPassword($password);

    /**
     * @return DateTime
     */
    public function getSubscribeDate();

    /**
     * @return DateTime
     */
    public function getLastUpdateDate();

    /**
     * @return DateTime
     */
    public function getLastLoginDate();

    /**
     * Restituisce la chiave di attivazione dell'utente
     *
     * @return string
     */
    public function getActivationKey();

    /**
     * Imposta la chiave di attivazione per l'utente
     *
     * @return $this
     */
    public function resetActivationKey();

    /**
     * Indica se l'utente è attivo, non attivo o disabilitato dall'amministratore
     *
     * @return bool|null
     */
    public function isActive();

    /**
     * Indica se l’utente è anonimo.
     *
     * @return bool
     */
    public function isAnonymous();

    /**
     * Indica se l’utente è amministratore.
     *
     * @return bool
     */
    public function isAdmin();

    /**
     * Indica se l’utente è founder.
     *
     * @return bool
     */
    public function isFounder();

    /**
     * Indica se l’utente è un cliente business.
     *
     * @return bool
     */
    public function isBusiness();

    /**
     * @return Role
     */
    public function getRole();

    /**
     * @return int
     */
    public function getRoleId();

    /**
     * @param int|Role $role
     * @return $this
     */
    public function setRole($role);

    /**
     * @param $key
     * @return bool
     */
    public function hasPermit($key);

    /**
     * @return bool
     */
    public function isDeletable();

    /**
     * Genera l'url alla creazione dell'utente.
     */
    public function generateUrl();

    /**
     * @return string
     */
    public function getUrl();

    /**
     * Attiva l'account
     *
     * @return $this
     */
    public function activate();

    /**
     * Disattiva l'account
     *
     * @return $this
     */
    public function deactivate();

    /**
     * @return UserDetailsInterface
     */
    public function getUserDetails();
}
