<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Localization;
use Application\Core\Model\LanguageSite;
use Application\Core\Model\Manager\AccountManager;
use Application\Core\Model\Manager\AccountManagerInterface;
use Application\Core\Model\Manager\ApplicationManager;
use Application\Core\Model\Manager\ApplicationManagerInterface;
use Application\Core\Model\Manager\CoreManagerRegistry;
use Application\Core\Model\Manager\CoreManagerRegistryCache;
use Application\Core\Model\Manager\GroupManager;
use Application\Core\Model\Manager\GroupManagerInterface;
use Application\Core\Model\Manager\LanguageManager;
use Application\Core\Model\Manager\LanguageModuleManager;
use Application\Core\Model\Manager\LanguageSiteManager;
use Application\Core\Model\Manager\ModuleManager;
use Application\Core\Model\Manager\ModuleSiteManager;
use Application\Core\Model\Manager\NotificationTemplateManager;
use Application\Core\Model\Manager\NotificationTemplateManagerInterface;
use Application\Core\Model\Manager\ProvinceManager;
use Application\Core\Model\Manager\ProvinceManagerInterface;
use Application\Core\Model\Manager\RegionManager;
use Application\Core\Model\Manager\RegionManagerInterface;
use Application\Core\Model\Manager\RoleManager;
use Application\Core\Model\Manager\RoleManagerInterface;
use Application\Core\Model\Manager\SettingManager;
use Application\Core\Model\Manager\SiteManager;
use Application\Core\Model\Site;
use Pongho\Core\LocalizationInterface;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

class ModelManagerProvider implements ServiceProviderInterface
{
    public function register(Container $container): void
    {
        $container->share('application_manager', function (): ApplicationManagerInterface {
            return new ApplicationManager();
        });

        $container->share('setting_manager', function (): SettingManager {
            return new SettingManager();
        });

        $container->share('account_manager', function (): AccountManagerInterface {
            return new AccountManager();
        });

        $container->share('region_manager', function (): RegionManagerInterface {
            return new RegionManager();
        });

        $container->share('province_manager', function (): ProvinceManagerInterface {
            return new ProvinceManager();
        });

        $container->share('role_manager', function (): RoleManagerInterface {
            return new RoleManager();
        });

        $container->share('group_manager', function (): GroupManagerInterface {
            return new GroupManager();
        });

        $container->share(
            'notification_template_manager',
            function (Container $container): NotificationTemplateManagerInterface {
                return new NotificationTemplateManager(
                    $container->get('site'),
                    $container->get('notification_collection'),
                );
            },
        );

        /**
         * Manager dei modelli base
         */
        $container->share('core_manager_registry', function (): CoreManagerRegistry {
            return new CoreManagerRegistry(new CoreManagerRegistryCache());
        });

        $container->set('site_manager', function (Container $container): SiteManager {
            return $container->get('core_manager_registry')->getSiteManager();
        });

        $container->set('language_manager', function (Container $container): LanguageManager {
            return $container->get('core_manager_registry')->getLanguageManager();
        });

        $container->set('module_manager', function (Container $container): ModuleManager {
            return $container->get('core_manager_registry')->getModuleManager();
        });

        $container->set('module_site_manager', function (Container $container): ModuleSiteManager {
            return $container->get('core_manager_registry')->getModuleSiteManager();
        });

        $container->set('language_site_manager', function (Container $container): LanguageSiteManager {
            return $container->get('core_manager_registry')->getLanguageSiteManager();
        });

        $container->set('language_module_manager', function (Container $container): LanguageModuleManager {
            return $container->get('core_manager_registry')->getLanguageModuleManager();
        });

        /**
         * Modelli base
         */
        $container->share('site', function (Container $container): Site {
            $siteId = $container->getParameter('site_id');

            if (PHP_SAPI !== 'cli') {
                /** @var \Pongho\Http\Request $request */
                $request = $container->get('request');

                if (
                    (
                        str_starts_with($request->getPathInfo(), '/pongho/')
                        || str_starts_with($request->getPathInfo(), '/api/')
                    )
                    && $request->query->has('site')
                ) {
                    $siteId = (int) $request->query->get('site');
                }
            }

            /** @var \Application\Core\Model\Manager\SiteManager $siteManager */
            $siteManager = $container->get('site_manager');

            $site = $siteManager->findById($siteId);

            if ($site === null) {
                throw new \RuntimeException('Site not found!');
            }

            return $site;
        });

        $container->share('main_site', function (Container $container): Site {
            try {
                $site = $container->get('site');
            } catch (\RuntimeException $runtimeException) {
                // Se non trovo il sito corrente, converto il messaggio dell’eccezione
                throw new \RuntimeException(
                    'Main site not found!',
                    $runtimeException->getCode(),
                    $runtimeException,
                );
            }

            if ($site->getId() !== 1) {
                /** @var \Application\Core\Model\Manager\SiteManager $siteManager */
                $siteManager = $container->get('site_manager');

                $site = $siteManager->findById(1);

                if ($site === null) {
                    throw new \RuntimeException('Main site not found!');
                }
            }

            return $site;
        });

        $container->share('language_site', function (Container $container): LanguageSite {
            if (PHP_SAPI === 'cli') {
                /** @var \Application\Core\Model\Site $site */
                $site = $container->get('site');

                return $site->getDefaultLanguage();
            } else {
                /**
                 * La lingua può essere decisa in base a 3 parametri:
                 *
                 * 1. l’indirizzo della pagina
                 * 2. le impostazioni dell'utente corrente
                 * 3. la lingua impostata nel browser
                 *
                 * Al terzo parametro ci si arriva solo se l’utente è anonimo. In questo caso, devo controllare
                 * se la lingua del browser è disponibile. In alternativa si utilizza quella di default del sito.
                 */

                // primo caso, controllo se la rotta ha impostato una lingua (salto se sono in amministrazione o nelle api)
                $request = $container->get('request');
                if (!str_starts_with((string) $request->getPathInfo(), '/pongho/')) {
                    $router = $container->get('router');
                    $parameters = $router->getParameters();

                    if (isset($parameters['site_language']) && $parameters['site_language'] instanceof LanguageSite) {
                        return $parameters['site_language'];
                    }
                }

                /**
                 * @var \Application\Core\User                              $user
                 * @var \Application\Core\Model\Site                        $site
                 * @var \Application\Core\Model\Manager\LanguageSiteManager $languageSiteManager
                 */

                // secondo e terzo caso, vediamo l’utente
                $user = $container->get('current_user');
                $site = $container->get('site');
                $languageSiteManager = $container->get('language_site_manager');

                /** @var \Application\Core\Model\Account $account */
                $account = $user->getAccount();

                $userLanguageSite = $languageSiteManager->findByLanguageAndSite(
                    $account->language_id,
                    $site->getId(),
                );

                if ($user->isLogged() && ($userLanguageSite !== null)) {
                    // secondo caso
                    return $userLanguageSite;
                } else {
                    // terzo caso
                    $request = $container->get('request');

                    $languages = [];
                    foreach ($site->getLanguages() as $languageSite) {
                        $languages[$languageSite->getLanguage()->getIso()] = $languageSite;
                    }

                    $preferred = $request->getPreferredLanguage(array_keys($languages));

                    if ($preferred && array_key_exists($preferred, $languages)) {
                        // lingua dal browser
                        return $languages[$preferred];
                    } else {
                        // lingua di default del sito
                        return $site->getDefaultLanguage();
                    }
                }
            }
        });

        $container->share('language', function (Container $container): LocalizationInterface {
            return new Localization($container->get('language_site'));
        });
    }
}
