<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Model\Site;
use Application\Core\User;
use DiscordHandler\DiscordHandler;
use Monolog\Formatter\HtmlFormatter;
use Monolog\Handler\BufferHandler;
use Monolog\Handler\FingersCrossedHandler;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SymfonyMailerHandler;
use Monolog\Logger;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Mailer\Mailer;
use Symfony\Component\Mime\Email;

class LoggerServiceProvider implements ServiceProviderInterface
{
    public function register(Container $container)
    {
        $container->share('logger', function (Container $container): LoggerInterface {
            $logger = new Logger('pongho');

            // handler
            $logger->pushHandler($container->get('logger.files_handler'));
            $logger->pushHandler($container->get('logger.discord_handler'));
            $logger->pushHandler($container->get('logger.mailer_handler'));

            // processor
            $logger->pushProcessor($container->get('logger.extra_processor'));

            return $logger;
        });

        $container->share('logger.files_handler', function (Container $container): HandlerInterface {
            $site = $container->get('site');

            return new RotatingFileHandler(
                $site->getContentPath(
                    $container->getParameter('logger.path', '/log') . '/pongho.log'
                ),
                $container->getParameter('logger.maxfiles', 90),
                Logger::DEBUG
            );
        });

        $container->share('logger.discord_handler', function (Container $container): HandlerInterface {
            $site = $container->get('site');

            // @link https://support.discord.com/hc/en-us/articles/228383668-Intro-to-Webhooks
            $webHooks = $container->getParameter(
                'logger.discord.webHooks',
                'https://discord.com/api/webhooks/1422575092604600341/xwbbf5UA93ZsoIy9wNSUpE82C3zFDec8ZnUNINWSjqKXJ_XacNGhiXoB_9qz16D_hdhJ'
            );

            $discordHandler = new DiscordHandler(
                $webHooks,
                $container->getParameter('logger.discord.name', $site->getName()),
                $container->getParameter('logger.discord.subName', $site->getDomain()),
                Logger::WARNING
            );

            $discordHandler->getConfig()->setEmbedMode();

            // In modalità "embed" non riporta i parametri "name" e "subName", quindi li aggiungo al messaggio
            $discordHandler->pushProcessor(function (array $record) use ($discordHandler) {
                $config = $discordHandler->getConfig();
                $record['message'] = sprintf('%s [%s] – %s', $config->getName(), $config->getSubName(), $record['message']);

                return $record;
            });

            return $discordHandler;
        });

        $container->share('logger.mailer_handler', function (Container $container): HandlerInterface {
            $mailer_handler = new SymfonyMailerHandler(
                $container->get(Mailer::class),
                $container->get('logger.mailer_message'),
                Logger::DEBUG
            );

            $mailer_handler->setFormatter(new HtmlFormatter());

            return new FingersCrossedHandler(
                new BufferHandler($mailer_handler),
                Logger::ERROR
            );
        });

        $container->share('logger.mailer_message', function (Container $container): Email {
            /** @var Site $site */
            $site = $container->get('site');

            $domain = $site->getDomain();
            $domain_info = parse_url($domain);
            $domain_host = $domain_info['host'];

            if (str_starts_with($domain_host, 'www.')) {
                $domain_host = substr($domain_host, 4);
            }

            return (new Email())
                ->from($container->getParameter('logger.mail.from', "no-reply@$domain_host"))
                ->to($container->getParameter('logger.mail.to', 'log@metaline.it'))
                ->subject($site->getName());
        });

        $container->share('logger.extra_processor', function (Container $container): callable {
            // Aggiunge il dominio del sito e l’ID utente nei log
            if (PHP_SAPI === 'cli') {
                $user_id = 'cli';
            } else {
                /** @var User $user */
                $user = $container->get('current_user');
                $user_id = $user->getAccount()->getId();
            }

            /** @var Site $site */
            $site = $container->get('site');

            return function (array $record) use ($site, $user_id) {
                $record['extra']['site_name'] = $site->getName();
                $record['extra']['site_domain'] = $site->getDomain();
                $record['extra']['user_id'] = $user_id;

                return $record;
            };
        });
    }
}
