<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use ActiveRecord\CacheWrapper;
use ActiveRecord\Config;
use ActiveRecord\Connection;
use ActiveRecord\Profiler\Profiler;
use Application\Core\Db\Maintainer\TaskList;
use Doctrine\DBAL\Configuration;
use Doctrine\DBAL\DriverManager;
use Doctrine\DBAL\Schema\DefaultSchemaManagerFactory;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * @refactoring Creare degli alias per i servizi del database in modo da rinominarli con:
 *   - db.connection
 *   - db.schema
 *   - db.profiler
 */
class DbServiceProvider implements ServiceProviderInterface
{
    public function register(Container $container): void
    {
        $container->share('db_profiler', function () {
            return new Profiler();
        });

        $container->share('connection', function (Container $container) {
            if ($container->get('kernel')->isDebug()) {
                // Attivo il trigger delle deprecazioni per Doctrine.
                $_ENV['DOCTRINE_DEPRECATIONS'] = 'trigger';
            }

            /**
             * @todo Creare un factory
             * @see BaseCase::loadFixture()
             */
            $params = Config::instance()->getDefaultConnection();

            $connectionParams = [
                'dbname'   => $params['database'],
                'user'     => $params['username'],
                'password' => $params['password'],
                'host'     => $params['host'],
                'charset'  => 'UTF8',
            ];

            if ('pgsql' === $params['adapter']) {
                $connectionParams['driver'] = 'pdo_pgsql';

                $driverVersion = Connection::instance()->query('SHOW server_version')->fetchColumn();
                if (is_string($driverVersion)) {
                    $connectionParams['serverVersion'] = $driverVersion;
                }
            } else {
                $connectionParams['driver'] = 'pdo_mysql';
            }

            $configuration = new Configuration();
            $configuration->setSchemaManagerFactory(new DefaultSchemaManagerFactory());

            return DriverManager::getConnection($connectionParams, $configuration);
        });

        $container->share('connection_schema', function (Container $container) {
            /** @var \Doctrine\DBAL\Connection $conn */
            $conn = $container->get('connection');

            return $conn->createSchemaManager()->introspectSchema();
        });

        $container->share('db.maintainer.task_list', function () {
            return new TaskList();
        });

        // Non sarebbe affar suo ma è per tenere tutto quello relativo al database nello stesso posto
        $this->registerOrm($container);
    }

    /**
     * Imposta la configurazione per la connessione al database.
     */
    protected function registerOrm(Container $container)
    {
        $path = $container->getParameter('orm.config_path');

        if (!file_exists($path)) {
            throw new \UnexpectedValueException(sprintf('Database configuration file not exists on path "%s"!', $path));
        }

        $settings = include $path;

        if (!array_key_exists('connections', $settings)) {
            throw new \UnexpectedValueException(sprintf('Connections not found on %s file.', $path));
        }

        if (!array_key_exists('default_connection', $settings)) {
            throw new \UnexpectedValueException(sprintf('Default connection not found on %s file.', $path));
        }

        $default_connection = $settings['default_connection'];

        if ($container->get('kernel')->getEnvironment() === 'test' && isset($settings['connections']['tests'])) {
            $default_connection = 'tests';
        }

        Config::instance()
            ->setConnections($settings['connections'])
            ->setDefaultConnectionName($default_connection)
            ->setDebug($container->get('kernel')->isDebug())
            ->setEventDispatcher($container->get('event_dispatcher'));

        if ($container->get('kernel')->isDebug()) {
            Config::instance()->setProfiler($container->get('db_profiler'));
        }

        try {
            Config::instance()->setCache(new CacheWrapper($container->get('cache')));
        } catch (\BadMethodCallException) {
            // do nothing
        }
    }
}
