<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Console\Command\ChangeDomainCommand;
use Application\Core\Console\Command\DbClearCommand;
use Application\Core\Console\Command\PhpStormMetaBuilderCommand;
use Application\Core\Export\DataExporter\UsersExporter;
use Application\Core\Export\DataExporterInterface;
use Application\Core\Export\Serializer\CsvSerializer;
use Application\Core\Export\SerializerInterface;
use Application\Core\I18n\Locale\it_IT\Locale;
use Application\Core\I18n\LocaleRegistry;
use Application\Core\Utilities\AccountInfoRenderer;
use Application\Core\Utilities\AdminUserNotificationHelper;
use Application\Core\Utilities\ChangeDomainHandlerFactory;
use Application\Core\Utilities\UserActivationLinkBuilder;
use ArrayObject;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

class CoreServiceProvider implements ServiceProviderInterface
{
    public function register(Container $container): void
    {
        $container->share(
            'user_activation_link_builder',
            function (): UserActivationLinkBuilder {
                return new UserActivationLinkBuilder();
            },
        );

        $container->share(
            'admin_user_notification_helper',
            function (Container $container): AdminUserNotificationHelper {
                return new AdminUserNotificationHelper(
                    $container->get('notification_sender'),
                    $container->get('notification_collection'),
                    $container->get('user_activation_link_builder'),
                );
            },
        );

        $container->share('account_info_renderer', function (Container $container): AccountInfoRenderer {
            return new AccountInfoRenderer($container->get('translator'));
        });

        //
        // I18n
        //

        $container->share('i18n.locale_registry', function (): LocaleRegistry {
            $registry = new LocaleRegistry();

            // fallback it_IT
            $registry->registerDefault(function () {
                return new Locale();
            });

            return $registry;
        });

        //
        // Export
        //

        $container->share('export.csv_serializer', function (): SerializerInterface {
            return new CsvSerializer();
        });

        //
        // Admin
        //

        $container->share('admin.export_serializer', function (Container $container): SerializerInterface {
            return $container->get('export.csv_serializer');
        });

        $container->share('admin.users.exporter', function (Container $container): DataExporterInterface {
            return new UsersExporter(
                $container->get('translator'),
                $container->get('connection'),
            );
        });

        //
        // Console
        //

        $container->share('core.change_domain_handler', function (Container $container) {
            return (new ChangeDomainHandlerFactory($container->get('connection')))->create();
        });

        $container->extend(
            'console_commands',
            function (ArrayObject $commands, Container $container): void {
                $commands->append(new ChangeDomainCommand($container->get('core.change_domain_handler')));
                $commands->append(new DbClearCommand($container->get('db.maintainer.task_list')));

                if ($container->get('kernel')->isDebug()) {
                    $commands->append(new PhpStormMetaBuilderCommand($container));
                }
            },
        );
    }
}
