<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Cron;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\ParameterType;

class TaskRepository
{
    /**
     * @var Connection
     */
    protected $connection;

    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * @param string $service_name
     * @return bool
     */
    public function create($service_name, array $data)
    {
        return (bool) $this->connection->insert(
            'pongho_cron_tasks',
            [
                'service_name' => $service_name,
                'data'         => json_encode($data),
                'created_at'   => new \DateTime(),
            ],
            [
                'service_name' => ParameterType::STRING,
                'data'         => ParameterType::STRING,
                'created_at'   => 'datetime',
            ]
        );
    }

    /**
     * @return \Doctrine\DBAL\Driver\Statement|Task[]
     * @throws \Doctrine\DBAL\Exception
     */
    public function allNext(): iterable
    {
        $sql = <<<SQL
SELECT *
  FROM pongho_cron_tasks
 WHERE executed_at IS NULL
   AND attempts < 5
   AND (last_attempt_at IS NULL OR last_attempt_at < :attempt_date)
 ORDER BY created_at
SQL;

        $sth = $this->connection->prepare($sql);
        $sth->bindValue('attempt_date', new \DateTime('10 minutes ago'), 'datetime');

        $result = $sth->executeQuery();

        foreach ($result->fetchAllAssociative() as $row) {
            yield Task::createFromArray($row);
        }
    }

    public function save(Task $task)
    {
        $r = new \ReflectionObject($task);

        $pk_fields = ['id'];
        $untouchable_fields = ['id', 'service_name', 'data', 'created_at'];

        $pk = [];
        $properties = [];

        foreach ($r->getProperties() as $property) {
            $name = $property->getName();

            if (in_array($name, $pk_fields)) {
                $pk[$name] = $property->getValue($task);
            }

            if (!in_array($name, $untouchable_fields)) {
                $properties[$name] = $property->getValue($task);
            }
        }

        $this->connection->update(
            'pongho_cron_tasks',
            $properties,
            $pk,
            [
                'id'              => ParameterType::INTEGER,
                'attempts'        => ParameterType::INTEGER,
                'last_attempt_at' => 'datetime',
                'executed_at'     => 'datetime',
            ]
        );
    }
}
