<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller;

use Application\Core\Controller;
use Application\Core\Model\Account;
use Application\Core\Model\Role;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\HandlerStack;
use kamermans\OAuth2\GrantType\AuthorizationCode;
use kamermans\OAuth2\OAuth2Middleware;
use kamermans\OAuth2\Signer\ClientCredentials\PostFormData;
use Pongho\GuzzleHttp\LoggerMiddleware;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Response;

final class LinkedInAuthController extends Controller
{
    /**
     * @route /user/linkedin-auth/
     */
    public function authAction()
    {
        if ($this->getHelper()->getUser()->isLogged()) {
            return $this->getRedirectAfterLoginResponse();
        }

        $authQuery = http_build_query([
            'response_type' => 'code',
            'client_id'     => $this->getClientId(),
            'redirect_uri'  => absolute_url('/user/linkedin-callback/'),
            'scope'         => $this->getAuthScopes(),
            'state'         => $this->prepareState(),
        ]);

        $authUrl = 'https://www.linkedin.com/oauth/v2/authorization?' . $authQuery;

        return new RedirectResponse($authUrl);
    }

    /**
     * @route /user/linkedin-callback/
     */
    public function callbackAction()
    {
        if ($this->getHelper()->getUser()->isLogged()) {
            return $this->getRedirectAfterLoginResponse();
        }

        $request = $this->getRequest();
        $logger = $this->getContainer()->get('logger');

        /*
         * Se l’applicazione viene rifiutata, rimando l’utente alla pagina di login per
         * provare ad accedere in un altro modo.
         *
         * @link https://docs.microsoft.com/en-us/linkedin/shared/authentication/authorization-code-flow?context=linkedin/context#application-is-rejected
         */
        $error = $request->query->get('error');

        if ($error) {
            if (in_array($error, ['user_cancelled_login', 'user_cancelled_authorize'])) {
                $logger->info(
                    '[Sign in with LinkedIn] Application rejected',
                    [
                        'request' => $request->query->all(),
                    ]
                );
            } else {
                $logger->error(
                    '[Sign in with LinkedIn] Login error',
                    [
                        'request' => $request->query->all(),
                    ]
                );
            }

            return $this->getHelper()->redirectToLogin();
        }

        /*
         * Se l’applicazione viene approvata, procedo al recupero dei dati dell’utente
         * da LinkedIn ed alla sua iscrizione al sito.
         *
         * @link https://docs.microsoft.com/en-us/linkedin/shared/authentication/authorization-code-flow?context=linkedin/context#your-application-is-approved
         */
        $code = $request->query->get('code');
        $state = $request->query->get('state');

        // Code & State parameters are mandatory
        if (!$code || !$state) {
            $logger->warning(
                '[Sign in with LinkedIn] No "code" or "state" parameters',
                [
                    'request' => $request->query->all(),
                ]
            );

            throw new HttpNotFoundException();
        }

        /*
         * Il parametro 'state' deve corrispondere al valore inviato nella prima richiesta.
         * Se i due valori non corrispondono, potrebbe trattarsi di un attacco di tipo CSRF.
         * In questo caso, l’applicazione dovrebbe restituire una risposta "401 Unauthorized".
         *
         * @link https://docs.microsoft.com/it-it/linkedin/shared/authentication/authorization-code-flow?context=linkedin/consumer/context#your-application-is-approved
         * @link https://en.wikipedia.org/wiki/Cross-site_request_forgery
         */
        if ($state !== $this->getState()) {
            $logger->error(
                '[Sign in with LinkedIn] The "state" parameter does not match',
                [
                    'request'      => $request->query->all(),
                    'requestState' => $state,
                    'sessionState' => $this->getState(),
                ]
            );

            throw new HttpUnauthorizedException();
        }

        try {
            /*
             * Creo l’istanza del client
             */
            $client = $this->createClient($code);

            /*
             * La prima richiesta serve per ottenere l’indirizzo e-mail, con il quale posso fare
             * un controllo sulla presenza o meno dell’utente.
             */
            $response = $client
                ->get('https://api.linkedin.com/v2/emailAddress?q=members&projection=(elements*(handle~))');
            $body = json_decode((string) $response->getBody(), true);

            if (isset($body['elements'][0]['handle~']['emailAddress'])) {
                $email = htmlspecialchars((string) $body['elements'][0]['handle~']['emailAddress']);
            } else {
                $email = '';
            }

            if (!$email) {
                throw new Exception('The "emailAddress" parameter is missing');
            }

            /*
             * Cerco l’account e se lo trovo forzo il login
             */
            $account = Account::findByEmail($email);

            if (!$account) {
                /*
                 * Nuovo utente.
                 */
                $response = $client->get('https://api.linkedin.com/v2/me');
                $body = json_decode((string) $response->getBody(), true);

                $firstName = isset($body['localizedFirstName']) ? htmlspecialchars((string) $body['localizedFirstName']) : '';
                $lastName = isset($body['localizedLastName']) ? htmlspecialchars((string) $body['localizedLastName']) : '';

                $attributes = $this->getHelper()->filter(
                    $this,
                    'sign_in_with_linkedin.account_attributes',
                    [
                        'language_id' => $this->getLanguageId(),
                        'role_id'     => Role::USER_LOGGED,
                        'is_active'   => true,
                        'is_founder'  => false,
                        'newsletter'  => true,
                        'privacy'     => true,
                        'email'       => $email,
                        'username'    => $email,
                        'password'    => random(8),
                        'name'        => $firstName,
                        'surname'     => $lastName,
                    ]
                );

                $account = Account::create($attributes);

                $this->getHelper()->notify(
                    $this,
                    'sign_in_with_linkedin.after_account_created',
                    [
                        'account' => $account,
                    ]
                );
            }

            $this->getHelper()->getUser()->forceLogin($account);

            $this->getHelper()->notify(
                $this,
                'sign_in_with_linkedin.after_login',
                [
                    'account' => $account,
                ]
            );

            return $this->getRedirectAfterLoginResponse();
        } catch (Exception $exception) {
            $logger->error(
                "[Sign in with LinkedIn] {$exception->getMessage()}",
                [
                    'exception' => $exception,
                ]
            );

            return $this->getHelper()->displayError(
                sprintf(
                    'An error has occurred. Please retry the procedure: %sLogin%s',
                    '<a href="' . $this->getHelper()->getLoginUrl() . '">',
                    '</a>'
                )
            );
        }
    }

    /**
     * @return string
     */
    private function getClientId()
    {
        return $this->getContainer()->get('site')->getOption('linkedin_client_id');
    }

    /**
     * @return string
     */
    private function getClientSecret()
    {
        return $this->getContainer()->get('site')->getOption('linkedin_client_secret');
    }

    /**
     * @return string
     */
    private function getAuthScopes()
    {
        return implode(' ', $this->getHelper()->filter(null, 'sign_in_with_linkedin.scopes', [
            'r_emailaddress',
            'r_liteprofile',
        ]));
    }

    /**
     * @return string
     */
    private function prepareState()
    {
        $state = md5(random());

        $this->getContainer()->get('session')->set('linkedin-state', $state);

        return $state;
    }

    /**
     * @return string
     */
    private function getState()
    {
        return (string) $this->getContainer()->get('session')->get('linkedin-state');
    }

    /**
     * @param string $code
     * @return Client
     */
    private function createClient($code)
    {
        $stack = HandlerStack::create();
        $stack->push($this->createLoggerMiddleware());
        $stack->push($this->createAuthMiddleware($code));

        return new Client([
            'auth'    => 'oauth',
            'handler' => $stack,
        ]);
    }

    /**
     * @return LoggerMiddleware
     */
    private function createLoggerMiddleware()
    {
        return new LoggerMiddleware($this->getContainer()->get('logger'), 'Sign in with LinkedIn');
    }

    /**
     * @param string $code
     * @return OAuth2Middleware
     */
    private function createAuthMiddleware($code)
    {
        $stack = HandlerStack::create();
        $stack->push($this->createLoggerMiddleware());

        // @link https://docs.microsoft.com/en-us/linkedin/shared/authentication/authorization-code-flow?context=linkedin/context#step-3-exchange-authorization-code-for-an-access-token
        // @link https://github.com/kamermans/guzzle-oauth2-subscriber
        $reAuthClient = new Client([
            'base_uri' => 'https://www.linkedin.com/oauth/v2/accessToken',
            'handler'  => $stack,
        ]);

        $reAuthConfig = [
            'client_id'     => $this->getClientId(),
            'client_secret' => $this->getClientSecret(),
            'code'          => $code,
            'redirect_uri'  => absolute_url('/user/linkedin-callback/'),
            'scope'         => $this->getAuthScopes(),
        ];

        $grantType = new AuthorizationCode($reAuthClient, $reAuthConfig);
        $oauth = new OAuth2Middleware($grantType);
        $oauth->setClientCredentialsSigner(new PostFormData());

        // @todo: Devo salvare il token di accesso?
        // $this->accessToken = $oauth->getAccessToken();

        return $oauth;
    }

    /**
     * @return Response
     */
    private function getRedirectAfterLoginResponse()
    {
        return $this->getHelper()->redirectResponse(
            $this->getHelper()->filter(
                $this,
                'sign_in_with_linkedin.redirect_after_login',
                '/'
            )
        );
    }
}
