<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\Field\TableChecklistField;
use Application\Admin\Form\FormConfig;
use Application\Core\Entity\AccountInterface;
use Application\Core\Model\Account;
use Application\Core\Model\AccountPortfolio;
use Application\Core\Model\File;
use Application\Core\Model\Group;
use Application\Core\Model\GroupUser;
use Application\Core\Model\Language;
use Application\Core\Model\Manager\AccountManager;
use Application\Core\Model\Permit;
use Application\Core\Model\Province;
use Application\Core\Model\Role;
use Monolog\Logger;
use Pongho\Form\Field;
use Pongho\Form\Field\SelectField;
use Pongho\Form\Filter\EmailFilter;
use Pongho\Form\Form;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\JsonResponse;
use Pongho\Http\RedirectResponse;
use Pongho\Http\Response;
use Pongho\Template\View;
use Pongho\Template\ViewClosure;
use Pongho\Utilities\Inflector;

class UsersController extends CrudFormController
{
    private $language_count;

    /**
     * Variabile usata per conservare la password in chiaro in caso di notifica all'utente
     *
     * @var string
     */
    protected $password;

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return \Application\Core\Model\Account::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        if ($this->getAction() === 'add') {
            return $this->getHelper()->getTranslator()->trans('Add a new user');
        }

        return $this->getHelper()->getTranslator()->trans('Edit user');
    }

    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'users';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.users.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getHelper()->getTranslator()->trans('Users list');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getHelper()->getTranslator()->trans('Add a new user');
    }

    /**
     * {@inheritdoc}
     */
    protected function hasAddButton()
    {
        return $this->getHelper()->userHasPermit('users.add');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return $this->getHelper()->getTranslator()->trans('Add the first user');
    }

    /**
     * {@inheritdoc}
     */
    protected function getMassActions()
    {
        if ($this->getHelper()->userHasPermit('users.delete')) {
            return parent::getMassActions();
        }

        return [];
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelOptions()
    {
        if ($this->getHelper()->getUser()->isFounder()) {
            $model_options = [
                'select'     => '`from`.*',
                'conditions' => [
                    '`from`.id <> :anonymous',
                    'anonymous' => Account::ANONYMOUS,
                ],
            ];
        } elseif (!Permit::isEnabled('users.view_all_users') || $this->getHelper()->userHasPermit('users.view_all_users')) {
            $model_options = [
                'select'     => '`from`.*',
                'conditions' => [
                    '`from`.id <> :anonymous AND `from`.is_founder <> :is_founder',
                    'anonymous'  => Account::ANONYMOUS,
                    'is_founder' => true,
                ],
            ];
        } else {
            $model_options = [
                'select'     => '`from`.*',
                'joins'      => 'LEFT JOIN ' . AccountPortfolio::tableName() . ' AS p ON p.descendant_id = `from`.id',
                'conditions' => [
                    '`from`.id <> :anonymous AND `from`.is_founder <> :is_founder AND p.ancestor_id = :ancestor',
                    'anonymous'  => Account::ANONYMOUS,
                    'is_founder' => true,
                    'ancestor'   => $this->getHelper()->getUserId(),
                ],
            ];
        }

        $model_options['from'] = Account::tableName() . ' AS `from`';
        $model_options['include'] = ['role'];

        /** @var \Application\Core\Model\Manager\ApplicationManager $app_manager */
        $app_manager = $this->getContainer()->get('application_manager');

        if ($app_manager->isInstalled('Showcase')) {
            $model_options['include'][] = 'total_orders';
        }

        // Ordinamento per data di creazione, dal più recente al più vecchio
        $model_options['order'] = 'created_at DESC';

        return $model_options;
    }

    /**
     * {@inheritdoc}
     */
    protected function getSearchFields()
    {
        return ['name', 'surname', 'fullname', 'email', 'company'];
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        $translator = $this->getHelper()->getTranslator();

        return [
            [
                'name'      => 'fullname',
                'label'     => $translator->trans('Name'),
                'class'     => 'main',
                'orderable' => true,
                'render'    => function (Account $account) {
                    return '<strong>' . $account->name() . '</strong><div class="actions"><span>' . $account->actions . '</span></div>';
                },
            ],
            [
                'name'      => 'role_id',
                'label'     => 'Ruolo',
                'class'     => 'large',
                'orderable' => true,
                'render'    => function (Account $account) {
                    return $account->getRole()->getName();
                },
            ],
            [
                'name'      => 'company',
                'label'     => $translator->trans('Company'),
                'class'     => 'large',
                'orderable' => true,
            ],
            [
                'name'      => 'username',
                'label'     => $translator->trans('Username'),
                'class'     => 'large',
                'orderable' => true,
            ],
            [
                'name'      => 'email',
                'label'     => $translator->trans('E-Mail address'),
                'class'     => 'large',
                'orderable' => true,
            ],
            [
                'name'      => 'created_at',
                'label'     => $translator->trans('Subscription date'),
                'orderable' => true,
                'render'    => function (Account $account) {
                    return $account->created_at->format('d-m-Y');
                },
            ],
            [
                'name'      => 'id',
                'label'     => 'ID',
                'class'     => 'id',
                'orderable' => true,
            ],
        ];
    }

    /**
     * {@inheritdoc}
     *
     * @param Account $account
     */
    public function parseArchiveRow($account)
    {
        $translator = $this->getHelper()->getTranslator();
        $path = $this->getHelper()->getPath();

        $actions = [];

        if ($this->getHelper()->userHasPermit('users.view')) {
            $actions[] = [
                $translator->trans('View'),
                'href'  => $this->url("/{$path}view/{$account->getId()}/"),
                'class' => 'view',
            ];
        }

        if ($this->getHelper()->userHasPermit('users.edit')) {
            $actions[] = [
                $translator->trans('Edit'),
                'href'  => $this->url("/{$path}edit/{$account->getId()}/"),
                'class' => 'edit',
            ];
        }

        if ($this->getHelper()->userHasPermit('users.delete')
            && $account->getId() != $this->getHelper()->getUserId()
            && $account->isDeletable()
        ) {
            $actions[] = [
                $translator->trans('Delete'),
                'href'  => $this->url("/{$path}delete/{$account->getId()}/"),
                'class' => 'delete',
            ];
        }

        $account->actions = $this->parseActions($actions);
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelFieldsConfig()
    {
        $translator = $this->getHelper()->getTranslator();

        $config = [
            'name'           => ['label' => $translator->trans('Name')],
            'surname'        => ['label' => $translator->trans('Surname')],
            'username'       => ['label' => $translator->trans('Username')],
            'email'          => ['label' => $translator->trans('Email')],
            'created_at'     => ['label' => $translator->trans('Subscribe date')],
            'language_id'    => ['label' => $translator->trans('Language'), 'data-type' => 'foreign'],
            'role_id'        => ['label' => $translator->trans('Role'), 'data-type' => 'foreign'],
            'group'          => [
                'label'     => $translator->trans('Group'),
                'data-type' => 'foreign',
                'column'    => 'g.group_id',
                'join'      => 'LEFT JOIN ' . GroupUser::tableName() . ' AS g ON g.user_id = `from`.id',
            ],
            'address'        => ['label' => $translator->trans('Address')],
            'address2'       => ['label' => $translator->trans('Address info')],
            'city'           => ['label' => $translator->trans('City')],
            'postcode'       => ['label' => $translator->trans('Post code')],
            'province_id'    => ['label' => $translator->trans('Province'), 'data-type' => 'foreign'],
            'company'        => ['label' => $translator->trans('Company')],
            'fax'            => ['label' => $translator->trans('Fax')],
            'telephone'      => ['label' => $translator->trans('Telephone')],
            'partita_iva'    => ['label' => $translator->trans('VAT code')],
            'codice_fiscale' => ['label' => $translator->trans('Tax code')],
            'sdi_code'       => ['label' => $translator->trans('SdI code')],
            'pec'            => ['label' => $translator->trans('PEC')],
            'is_active'      => ['label' => $translator->trans('User status'), 'data-type' => 'enum'],
        ];

        /** @var \Application\Core\Model\Manager\ApplicationManager $app_manager */
        $app_manager = $this->getContainer()->get('application_manager');

        if ($app_manager->isEnabled('Newsletter', $this->getHelper()->getSiteId())) {
            $config['newsletter'] = ['label' => $translator->trans('Newsletter')];
        }

        return $config;
    }

    /**
     * {@inheritdoc}
     */
    protected function getFilterEnumValues()
    {
        $translator = $this->getHelper()->getTranslator();

        return [
            'is_active'  => [
                'F'    => $translator->trans('Not active'),
                'T'    => $translator->trans('Active'),
                'null' => $translator->trans('Pending activation'),
            ],
            'newsletter' => [
                'F' => $translator->trans('Not active'),
                'T' => $translator->trans('Active'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    protected function getFilterSelectValues()
    {
        /** @var \Pongho\Template\Html $html */
        $html = $this->getContainer()->get('template_html');

        return [
            'language_id' => function () {
                $options = [
                    'select' => 'id, name',
                    'order'  => 'name ASC',
                ];

                $o = [];

                /** @var Language $row */
                foreach (Language::all($options) as $row) {
                    $o[$row->id] = $row->name;
                }

                return $o;
            },
            'province_id' => function () use ($html) {
                $options = [
                    'select' => 'id, name, plate',
                    'order'  => 'name ASC',
                ];

                $o = [];

                /** @var Province $row */
                foreach (Province::all($options) as $row) {
                    $o[$row->id] = $row->name;
                }

                return $html->select($o);
            },
            'role_id'     => function () {
                $options = [
                    'select'     => 'id, name',
                    'conditions' => ['id <> ?', Account::ANONYMOUS],
                    'order'      => 'position ASC, id ASC',
                ];

                $o = [];

                /** @var Role $row */
                foreach (Role::all($options) as $row) {
                    $o[$row->id] = $row->name;
                }

                return $o;
            },
            'group'       => function () {
                $options = [
                    'select' => 'id, name',
                    'order'  => 'position ASC, name ASC',
                ];

                $o = [];

                /** @var Group $row */
                foreach (Group::all($options) as $row) {
                    $o[$row->id] = $row->name;
                }

                return $o;
            },
        ];
    }

    /**
     * Azione "view"
     */
    public function viewAction()
    {
        if (!$this->getHelper()->userHasPermit('users.view')) {
            throw new HttpUnauthorizedException();
        }

        $form = $this->getForm();
        $form->setReadonly(true);
        $form->remove('notifications');
        $form->remove('actions');

        $this->renderAddEdit($form);
    }

    /**
     * @return int
     */
    private function getLanguageCount()
    {
        if ($this->language_count === null) {
            $this->language_count = Language::count();
        }

        return $this->language_count;
    }

    /**
     * {@inheritdoc}
     */
    protected function isSubmit()
    {
        return ($this->getRequest()->getMethod() === 'POST') || $this->isNotify() || $this->isCredentialsNotify();
    }

    /**
     * Restituisce true se devo inviare la notifica al cliente. Altrimenti false.
     *
     * @return boolean
     */
    protected function isNotify()
    {
        $form_request = $this->getRequest()->post->get('user', []);

        return (isset($form_request['notify_edit']) && $form_request['notify_edit'] === 'on');
    }

    /**
     * Restituisce true se devo comunicare delle nuove credenziali di accesso all'utente
     *
     * @return bool
     */
    protected function isCredentialsNotify()
    {
        $form_request = $this->getRequest()->post->get('user', []);

        return (isset($form_request['notify_credentials']) && $form_request['notify_credentials'] === 'on');
    }

    /**
     * Restituisce true se devo comunicare un nuovo link di attivazione all'utente
     *
     * @return bool
     */
    protected function isActivationLinkNotify()
    {
        $form_request = $this->getRequest()->post->get('user', []);

        return (isset($form_request['notify_activation_link']) && $form_request['notify_activation_link'] === 'on');
    }

    /**
     * @return bool
     */
    protected function isCreationNotify()
    {
        $form_request = $this->getRequest()->post->get('user', []);

        return (isset($form_request['notify_create']) && $form_request['notify_create'] === 'on');
    }

    /**
     * @return bool
     */
    protected function isStatusChangeNotify()
    {
        $form_request = $this->getRequest()->post->get('user', []);

        return (isset($form_request['notify_status']) && $form_request['notify_status'] === 'on');
    }

    /**
     * @return bool
     */
    protected function isActivationNotify()
    {
        return ($this->isStatusChangeNotify() && $this->getModel()->isActive() === true);
    }

    /**
     * @return bool
     */
    protected function isDeactivationNotify()
    {
        return ($this->isStatusChangeNotify() && $this->getModel()->isActive() === false);
    }

    /**
     * Callback before_update
     */
    protected function beforeUpdateModelCallback(Account $account, array $fields)
    {
        $this->getHelper()->notify(
            $this,
            $this->getEventNamespace('before_update_model'),
            ['account' => $account, 'fields' => $fields]
        );
    }

    /**
     * Callback before_save
     */
    public function beforeSaveCallback(Form $form)
    {
        /** @var \Application\Core\Model\Account $account */
        $account = $form->getSubject()->getModel();

        // Serve per la validazione del modello
        $account->privacy = true;

        // Faccio in modo di non poter disattivare se stessi
        if ($account->id == $this->getHelper()->getUserId()) {
            $account->activate();
        }

        $this->getHelper()->notify($this, $this->getEventNamespace('before_save'), ['account' => $account]);
    }

    /**
     * Callback after_save
     */
    protected function afterSaveCallback(Form $form)
    {
        /** @var \Application\Core\Model\Account $account */
        $account = $form->getSubject()->getModel();

        // Invio la notifica se ho selezionato qualche spunta.
        $this->sendNotifications($account);

        $this->getHelper()->notify($this, $this->getEventNamespace('after_save'), ['account' => $account]);
    }

    /**
     * @param              $value
     */
    public function setPasswordCallback(ModelSubject $subject, $value)
    {
        // Se sto inviando una notifica all'utente
        if (empty($value)) {
            /** @var \Application\Core\Model\Site $site */
            $site = $this->getContainer()->get('main_site');

            if (
                $this->isCredentialsNotify()
                || $this->isCreationNotify()
                || ($site->getOption('subscription_mode') === 'admin' && $this->isStatusChangeNotify())
            ) {
                $value = random(8);
            }
        }

        // Backup della password in chiaro da usare per le notifiche
        $this->password = $value;

        /** @var \Application\Core\Model\Account $account */
        $account = $subject->getModel();

        $account->password = $value;
        $account->confirm_password = $value;
    }

    /**
     * @param              $value
     * @throws \UnexpectedValueException
     */
    public function setAvatarCallback(ModelSubject $subject, $value)
    {
        if (!is_array($value)) {
            throw new \UnexpectedValueException('The value returned for the "avatar" field must be an array');
        }

        /** @var \Application\Core\Model\Site $site */
        $site = $this->getContainer()->get('main_site');

        $value['site_id'] = $site->id;
        $value['path'] = substr((string) $value['path'], strlen($site->getUploadsPath()));

        /** @var Account $account */
        $account = $subject->getModel();

        if (is_numeric($file_id = $account->avatar_id)) {
            $file = File::find($file_id);

            $account->avatar_id = null;
            $account->save(false);

            $file->delete();
        }

        /** @var File $file */
        $file = File::create($value);

        $account->avatar_id = $file->id;
    }

    /**
     * @return array
     */
    public function getGroupsCallback(ModelSubject $subject)
    {
        /** @var Account $model */
        $model = $subject->getModel();

        return array_keys($model->getGroups());
    }

    /**
     * @param              $value
     */
    public function setGroupsCallback(ModelSubject $subject, $value)
    {
        /** @var Account $model */
        $model = $subject->getModel();

        $model->setGroups($value);
    }

    /**
     * Callback per il subject della form.
     *
     * Restituisce l’elenco degli ID dei genitori diretti (primo livello) dell’utente.
     *
     * @return array
     */
    public function getSubjectParentsCallback(ModelSubject $subject)
    {
        /** @var \Application\Core\Model\Account $model */
        $model = $subject->getModel();

        $parents = [];
        foreach ($model->getParents() as $parent) {
            $parents[] = $parent->id;
        }

        return $parents;
    }

    /**
     * @param              $value
     */
    public function setSubjectParentsCallback(ModelSubject $subject, $value)
    {
        /** @var \Application\Core\Model\Account $model */
        $model = $subject->getModel();

        $model->setParents($value);
    }

    /**
     * @param bool|null $status
     * @return string
     */
    protected function accountIsActiveToString($status)
    {
        if ($status === null) {
            return 'pending';
        }

        if ($status === false) {
            return 'not active';
        }

        if ($status === true) {
            return 'active';
        }

        throw new \InvalidArgumentException('The value "' . $status . '" is not valid');
    }

    /**
     * @param string $status
     * @return bool|null
     */
    protected function accountIsActiveFromString($status)
    {
        if ($status === 'pending') {
            return null;
        }

        if ($status === 'not active') {
            return false;
        }

        if ($status === 'active') {
            return true;
        }

        throw new \InvalidArgumentException('The value "' . $status . '" is not valid');
    }

    /**
     * @param ModelSubject $subject
     * @return string
     */
    public function getSubjectIsActiveCallback($subject)
    {
        /** @var AccountInterface $model */
        $model = $subject->getModel();

        return $this->accountIsActiveToString($model->isActive());
    }

    /**
     * @param              $value
     */
    public function setSubjectIsActiveCallback(ModelSubject $subject, $value)
    {
        /** @var Account $model */
        $model = $subject->getModel();

        $value = $this->accountIsActiveFromString($value);

        if (!$model->isActive() && $value) {
            $this->getHelper()->notify($this, 'account_activation', ['account' => $subject->getModel()]);
        }

        $model->is_active = $value;
    }

    /**
     * @var Account
     */
    protected $account;

    /**
     * {@inheritdoc}
     */
    protected function getModel()
    {
        if ($this->account === null) {
            $this->account = parent::getModel();

            if ($this->getAction() == 'add') {
                $site = $this->getHelper()->getSite();

                $this->account->language_id = $site->default_language_id;
                $this->account->password = random(8);
                $this->account->is_active = $site->getOption('subscription_mode') !== 'email' ? true : null;
                $this->account->newsletter = true;
            }
        }

        return $this->account;
    }

    /**
     * Restituisce il Subject per la Form
     *
     * @return \Pongho\Form\Subject\ModelSubject
     */
    protected function getSubject()
    {
        $class = $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject_class'), \Pongho\Form\Subject\ModelSubject::class);

        /** @var \Pongho\Form\Subject\ModelSubject $subject */
        $subject = new $class($this->getModel());

        // Collego la gestione dei tipi di campo particolari al subject
        $subject
            ->attachSetter('password', $this->setPasswordCallback(...))
            ->attachSetter('avatar_id', $this->setAvatarCallback(...))

            ->attachGetter('groups', $this->getGroupsCallback(...))
            ->attachSetter('groups', $this->setGroupsCallback(...))

            ->attachGetter('ancestors', $this->getSubjectParentsCallback(...))
            ->attachSetter('ancestors', $this->setSubjectParentsCallback(...))

            ->attachGetter('is_active', $this->getSubjectIsActiveCallback(...))
            ->attachSetter('is_active', $this->setSubjectIsActiveCallback(...));

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject'), $subject);
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        return $this->getHelper()->filter(
            $this,
            $this->getEventNamespace('filter_form_options'),
            $this->getDefaultFormsOptions(),
            [
                'subject' => $this->getSubject(),
            ]
        );
    }

    /**
     * Restituisce le opzioni di base della form.
     *
     * @return \Pongho\Form\Utilities\FormConfig
     */
    protected function getDefaultFormsOptions()
    {
        /** @var \Pongho\Form\Subject\ModelSubject $subject */
        $subject = $this->getSubject();

        /** @var \Application\Core\Model\Account $account */
        $account = $subject->getModel();

        /** @var \Application\Core\Model\Site $site */
        $site = $this->getContainer()->get('main_site');

        $translator = $this->getHelper()->getTranslator();

        // Gestione del campo password
        $password_field = [
            'class'      => \Pongho\Form\Field\PasswordField::class,
            'label'      => $translator->trans('Password'),
            'attributes' => [
                'class'        => 'input_text',
                'autocomplete' => 'new-password',
                'placeholder'  => $translator->trans('Fill out only for change'),
            ],
        ];

        if ($this->action === 'add') {
            $password_field['attributes']['required'] = 'required';
            unset($password_field['attributes']['placeholder']);
        }

        // Gestione del campo language
        if ($this->getLanguageCount() > 1) {
            $language_field = [
                'class' => \Application\Core\Form\Field\LanguageField::class,
                'label' => $translator->trans('Language'),
            ];
        } else {
            $language_field = ['class' => \Pongho\Form\Field\HiddenField::class];
        }

        $config = new FormConfig('user', $subject, $this->getHelper()->getLocalization());

        $config->addBaseStructure($this->getActiveTab(), $this->currentUrl());

        // Pannello informazioni utente
        $config->addPanelBefore(
            'content',
            'info',
            null,
            false,
            ['view' => new ViewClosure($this->getAccountInfoView(...))]
        );

        // Pannello notifiche
        $config->addPanelBefore(
            'actions',
            'notifications',
            'Notifiche',
            false,
            [
                'description' => $translator->trans(
                    'Will be sent an email for each selected notification. The notifications can be sent only to active
                     accounts or in the activation phase, with the exception of the notification of deactivation. Some
                     notifications can be selected only after certain operations.'
                ),
            ]
        );

        if ($this->getAction() === 'edit') {
            $attributes = [];

            if (!$account->isActive()) {
                $attributes = [
                    'disabled' => 'disabled',
                ];
            }

            if ($account->isActive() === null && $site->getOption('subscription_mode') === 'email') {
                $config->addField(
                    'notifications/notify_activation_link',
                    [
                        'class'       => \Pongho\Form\Field\CheckboxField::class,
                        'label'       => $translator->trans('Activation link'),
                        'description' => $translator->trans(
                            'Will be sent a new single use activation link'
                        ),
                    ]
                );
            }

            $config->addField(
                'notifications/notify_edit',
                [
                    'class'       => \Pongho\Form\Field\CheckboxField::class,
                    'label'       => $translator->trans('Account edit'),
                    'description' => $translator->trans(
                        'Will not list the effective changes, it only notify the user about the change'
                    ),
                    'attributes'  => $attributes,
                ]
            );

            $config->addField(
                'notifications/notify_status',
                [
                    'class'       => \Pongho\Form\Field\CheckboxField::class,
                    'label'       => $translator->trans('Status change'),
                    'description' => $translator->trans(
                        'Will be sent a notify about the status change'
                    ),
                    'attributes'  => [
                        'disabled' => 'disabled',
                    ],
                ]
            );

            $config->addField(
                'notifications/notify_credentials',
                [
                    'class'       => \Pongho\Form\Field\CheckboxField::class,
                    'label'       => $translator->trans('New access credentials'),
                    'description' => $translator->trans(
                        'Will be sent the new password entered in the field or a random one if left empty'
                    ),
                    'attributes'  => $attributes,
                ]
            );
        } else {
            if ($site->getOption('subscription_mode') === 'email') {
                $description = $translator->trans(
                    'Will be sent the credentials, and the activation link if applicable, for the new account'
                );
            } else {
                $description = $translator->trans(
                    'Will be sent the credentials for the new account'
                );
            }

            $config->addField(
                'notifications/notify_create',
                [
                    'class'       => \Pongho\Form\Field\CheckboxField::class,
                    'label'       => $translator->trans('Account creation'),
                    'description' => $description,
                ]
            );
        }

        $config->addField(
            'content/main',
            [
                'class' => \Application\Admin\Form\PanelFieldset::class,
                'label' => $translator->trans('Main'),
            ]
        );

        $config->addFields(
            'content/main',
            [
                'account' => [
                    'class'      => \Pongho\Form\Fieldset::class,
                    'attributes' => ['class' => 'panel'],
                    'label'      => $translator->trans('Account'),
                ],
                'profile' => [
                    'class'      => \Pongho\Form\Fieldset::class,
                    'attributes' => ['class' => 'panel'],
                    'label'      => $translator->trans('Profile'),
                ],
            ]
        );

        $accountFields = [
            'username'    => [
                'class'      => \Pongho\Form\Field\TextField::class,
                'label'      => $translator->trans('Username'),
                'attributes' => [
                    'class'        => 'input_text',
                    'required'     => 'required',
                    'autocomplete' => 'off',
                ],
            ],
            'password'    => $password_field,
            'email'       => [
                'class'      => \Pongho\Form\Field\TextField::class,
                'label'      => $translator->trans('E-Mail address'),
                'attributes' => [
                    'class'        => 'input_text',
                    'required'     => 'required',
                    'autocomplete' => 'off',
                ],
                'filter'     => new EmailFilter(),
            ],
            'language_id' => $language_field,
            'role_id'     => [
                'class'   => \Pongho\Form\Field\SelectField::class,
                'label'   => $translator->trans('Role'),
                'options' => $this->getRolesOptionsCallback(...),
            ],
            'is_active'   => [
                'class'   => \Pongho\Form\Field\SelectField::class,
                'label'   => $translator->trans('User status'),
                'options' => $this->getStatusOptionsCallback(...),
            ],
        ];

        /** @var \Application\Core\Model\Manager\ApplicationManager $app_manager */
        $app_manager = $this->getContainer()->get('application_manager');

        if ($app_manager->isEnabled('Newsletter', $this->getHelper()->getSiteId())) {
            $accountFields['newsletter'] = [
                'class'       => \Pongho\Form\Field\CheckboxField::class,
                'label'       => $translator->trans('Newsletter'),
                'description' => $translator->trans('The user can receive the newsletter'),
            ];
        }

        $config->addFields('content/main/account', $accountFields);

        $config->addFields(
            'content/main/profile',
            [
                'avatar_id'   => [
                    'class'    => \Pongho\Form\Field\ImageField::class,
                    'label'    => $translator->trans('Avatar'),
                    'settings' => [
                        'delete_url'  => $this->url('/users/deleteavatar/' . $account->id . '/'),
                        'preview_url' => function () use ($account) {
                            if (($avatar = $account->getAvatar()) !== null) {
                                return $avatar->path();
                            }

                            return '';
                        },
                        'upload_path' => $site->getUploadsPath('/avatars/'),
                        //'allowed_extensions'	=> array(),
                        //'max_file_size'		=> null,
                        //'filename'			=> null,
                    ],
                ],
                'description' => [
                    'class' => \Pongho\Form\Field\TextareaField::class,
                    'label' => $translator->trans('Description'),
                ],
            ]
        );

        if ($this->getHelper()->getSite()->getOption('account_url') && $this->getContainer()->get('router')->has('user-view')) {
            $config->addFieldAfter(
                'content/main/account/email',
                'url',
                [
                    'class'       => \Application\Core\Form\Field\AccountUrlField::class,
                    'label'       => $translator->trans('Account public url'),
                    'description' => $translator->trans('Allows to change the url to access the user’s public profile'),
                    'settings'    => [
                        'url-path'       => $this->getAccountBaseUrl(),
                        'button-value'   => $translator->trans('Edit'),
                        'form-url'       => $this->url('/' . $this->getParameter('path') . '/changeurl/' . $account->id . '/'),
                        'lightbox-title' => $translator->trans('Edit the user’s url'),
                        'disable-edit'   => ($account->old_url !== ''),
                    ],
                ]
            );
        }

        $config->addFields(
            'content',
            [
                'details' => [
                    'class' => \Application\Admin\Form\PanelFieldset::class,
                    'label' => $translator->trans('Details'),
                ],
            ]
        );

        $config->addFields(
            'content/details',
            [
                'user-details' => [
                    'class'      => \Pongho\Form\Fieldset::class,
                    'attributes' => ['class' => 'panel'],
                    'label'      => $translator->trans('User data'),
                ],
                'user-address' => [
                    'class'      => \Pongho\Form\Fieldset::class,
                    'attributes' => ['class' => 'panel'],
                    'label'      => $translator->trans('Address'),
                ],
            ]
        );

        $config->addFields(
            'content/details/user-details',
            [
                'name'           => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Name'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'surname'        => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Surname'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'company'        => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Company'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'partita_iva'    => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('VAT'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'codice_fiscale' => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Tax code'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'sdi_code'       => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('SdI code'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'pec'            => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('PEC'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'telephone'      => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Telephone'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'fax'            => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Fax'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'website'        => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Website'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'googleplus'     => [
                    'class'       => \Pongho\Form\Field\TextField::class,
                    'label'       => $translator->trans('Google+ profile'),
                    'attributes'  => ['class' => 'input_text'],
                    'description' => $translator->trans('Enter the URL of your Google+ profile to connect the articles you have written on this site.'),
                ],
            ]
        );

        $config->addFields(
            'content/details/user-address',
            [
                'address'     => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Address'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'address2'    => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Address info'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'city'        => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('City'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'postcode'    => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Post code'),
                    'attributes' => ['class' => 'input_text'],
                ],
                'province_id' => [
                    'class'    => \Application\Core\Form\Field\ProvinceField::class,
                    'label'    => $translator->trans('Province'),
                    'settings' => [
                        'placeholder' => $translator->trans('Select a province'),
                    ],
                ],
            ]
        );

        // Visualizzo le altre form solo se l'azione non è add perché non posso salvare un nuovo utente dalla tab dei gruppi.
        if ($this->action !== 'add') {
            if ($this->getHelper()->userHasPermit('users.handle_groups')) {
                $groups = [];

                /** @var Group $group */
                foreach (Group::all(['order' => 'position ASC, name ASC']) as $group) {
                    $groups[$group->getId()] = '<strong>' . $group->getName() . '</strong><br><em>' . $group->getDescription() . '</em>';
                }

                if ($groups !== []) {
                    $config->addTab('content/groups', $translator->trans('Groups'));
                    $config->addPanel('content/groups/groups-panel');

                    $config->addField(
                        'content/groups/groups-panel/groups',
                        [
                            'class'    => TableChecklistField::class,
                            'options'  => $groups,
                            'settings' => [
                                'header_label' => $translator->trans('Group'),
                            ],
                        ]
                    );
                }
            }

            if ($this->getHelper()->userHasPermit('users.handle_assoc') && $this->getHelper()->getUserId() != $account->id) {
                $config->addTab('content/ancestors', $translator->trans('Ancestors'));
                $config->addPanel('content/ancestors/ancestors-panel');

                $options['ancestors'] = new FormConfig('details', $subject, $this->getHelper()->getLocalization());

                /** @var AccountManager $account_manager */
                $account_manager = $this->getContainer()->get('account_manager');

                $account_ancestors = $account_manager->findAncestorsByDescendant(
                    $account->getId(),
                    $this->getHelper()->getUserId(),
                    $this->getHelper()->userHasPermit('users.view_all_users')
                );

                /** @var \Application\Core\Model\Account $ancestor */
                $ancestors = [];
                foreach ($account_ancestors as $ancestor) {
                    $ancestors[$ancestor->id] = '<strong>' . $ancestor->name() . '</strong>';
                }

                $config->addField(
                    'content/ancestors/ancestors-panel/ancestors',
                    [
                        'class'    => \Application\Admin\Form\Field\TableChecklistField::class,
                        'options'  => $ancestors,
                        'settings' => [
                            'header_label' => $translator->trans('Ancestor'),
                        ],
                    ]
                );
            }
        }

        return $config;
    }

    /**
     * @return string
     */
    public function getAccountInfoView(Field $field)
    {
        /** @var Account $account */
        $account = $field->getForm()->getSubject()->getModel();

        /** @var \Application\Core\Utilities\AccountInfoRenderer $renderer */
        $renderer = $this->getContainer()->get('account_info_renderer');
        $renderer->setAccount($account);

        return $renderer->render();
    }

    /**
     * @return array
     *
     * @todo RoleManager
     */
    public function getRolesOptionsCallback()
    {
        $select_options = [];
        $query_options = [
            'conditions' => [
                'id <> :guest',
                'guest' => Role::USER_NOT_LOGGED,
            ],
            'order'      => 'position ASC, id ASC',
        ];

        /** @var Role $role */
        foreach (Role::all($query_options) as $role) {
            $select_options[$role->id] = $role->name;
        }

        return $select_options;
    }

    /**
     * @return array
     */
    public function getStatusOptionsCallback(SelectField $field)
    {
        /** @var Account $account */
        $account = $field->getForm()->getSubject()->getModel();

        $options = [];

        if ($account->isActive() === null) {
            $options['pending'] = $this->getHelper()->getTranslator()->trans('Pending activation');
        }

        $options['not active'] = $this->getHelper()->getTranslator()->trans('Not active');
        $options['active'] = $this->getHelper()->getTranslator()->trans('Active');

        return $options;
    }

    protected function sendNotifications(Account $account)
    {
        $this->prepareNotifications($account);

        $this->getNotificationHelper()->send();
    }

    protected function prepareNotifications(Account $account)
    {
        /** @var Logger $logger */
        $logger = $this->getContainer()->get('logger');

        $notification_helper = $this->getNotificationHelper();
        $admin_account = $this->getHelper()->getUser()->getAccount();

        $data = [
            'user'   => [
                'id'     => $account->getId(),
                'status' => $this->accountIsActiveToString($account->isActive()),
            ],
            'author' => [
                'id'       => $admin_account->getId(),
                'username' => $admin_account->getUsername(),
            ],
        ];

        if ($this->isNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Account edit notify', $data);

            if (!$notification_helper->addUserEditNotification($account)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore account edit notify', $data);
            }
        }

        if ($this->isStatusChangeNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Status change notify', $data);

            /** @var \Application\Core\Model\Site $site */
            $site = $this->getHelper()->getSite();

            if ($site->getOption('subscription_mode') === 'admin') {
                if (!$notification_helper->addUserActivatedNotification($account, $this->password)) {
                    $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore status change notify', $data);
                }
            } elseif (!$notification_helper->addUserStatusChangeNotification($account)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore status change notify', $data);
            }
        }

        if ($this->isCredentialsNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Send credentials notify', $data);

            if (!$notification_helper->addUserCredentialsNotification($account, $this->password)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore send credentials notify', $data);
            }
        }

        if ($this->isActivationLinkNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Send activation link notify', $data);

            if (!$notification_helper->addUserActivationLinkNotification($account)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore send activation link notify', $data);
            }
        }

        if ($this->isCreationNotify()) {
            $logger->debug('[' . strtoupper($this->getNamespace()) . '] Account create notify', $data);

            if (!$notification_helper->addUserCreationNotification($account, $this->password)) {
                $logger->debug('[' . strtoupper($this->getNamespace()) . '] Ignore account create notify', $data);
            }
        }
    }

    /**
     * Restituisce l’aiutante per le modifiche.
     *
     * Sovrascrivendo questo metodo è possibile sostituire l’aiutante con uno scritto ad hoc.
     *
     * @return \Application\Core\Utilities\AdminUserNotificationHelper
     */
    protected function getNotificationHelper()
    {
        return $this->getContainer()->get('admin_user_notification_helper');
    }

    /**
     * @return string
     * @deprecated Usare il servizio "user_activation_link_builder"
     */
    protected function getUserActivationLink(AccountInterface $account)
    {
        /** @var \Application\Core\Utilities\UserActivationLinkBuilder $builder */
        $builder = $this->getContainer()->get('user_activation_link_builder');

        return $builder->build($account);
    }

    /**
     * @return JsonResponse|RedirectResponse
     * @todo deprecare in funzione di un metodo comune a tutti i campi file da mettere in AdminController
     */
    public function deleteavatarAction()
    {
        $id = $this->getParameter('id', null, true);

        /** @var AccountManager $account_manager */
        $account_manager = $this->getContainer()->get('account_manager');
        $account = $account_manager->findById($id);

        if ($account) {
            $account->deleteAvatar();
        }

        if ($this->getRequest()->isAjax()) {
            return $this->getHelper()->displayJsonMessage(
                $this->getHelper()->getTranslator()->trans('Avatar deleted')
            );
        }

        return new RedirectResponse($this->url(sprintf('/%s/edit/%d/', $this->getParameter('path'), $id)));
    }

    /**
     * {@inheritdoc}
     */
    protected function checkAddPermit()
    {
        if (!$this->getHelper()->userHasPermit('users.add')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function checkEditPermit()
    {
        $this->checkUserEditPermit();
        $this->checkPortfolioRelationPermit();
    }

    /**
     * {@inheritdoc}
     */
    protected function checkDeletePermit()
    {
        $this->checkUserDeletePermit();
        $this->checkPortfolioRelationPermit();
    }

    /**
     * @throws HttpUnauthorizedException
     */
    protected function checkUserEditPermit()
    {
        if (!$this->getHelper()->userHasPermit('users.edit')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * @throws HttpUnauthorizedException
     */
    protected function checkUserDeletePermit()
    {
        if (!$this->getHelper()->userHasPermit('users.delete')) {
            throw new HttpUnauthorizedException();
        }
    }

    /**
     * @throws HttpUnauthorizedException
     */
    protected function checkPortfolioRelationPermit()
    {
        if ($this->getHelper()->userHasPermit('users.is_ancestor') && !$this->getHelper()->userHasPermit('users.view_all_users')) {
            $account = $this->getModel();

            if (!$this->getHelper()->getUser()->getAccount()->hasDescendant($account->id)) {
                // Eseguo un controllo di coerenza, nel caso l'utente corrente non riesca a vedersi perché manca la relazione
                if ($this->getHelper()->getUserId() === $account->id) {
                    throw new \LogicException(sprintf(
                        'The relation for the user %s with himself is missing, check the database',
                        $this->getHelper()->getUserId()
                    ));
                }

                throw new HttpUnauthorizedException();
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function renderAddEdit(Form $form)
    {
        parent::renderAddEdit($form);

        $this->getHelper()
            ->addCss(pongho_url(
                '/Application/Core/Resources/views/css/style.css?v='
                . filemtime(PONGHO_PATH . '/Application/Core/Resources/views/css/style.css')
            ))
            ->addJavaScript(pongho_url(
                '/Application/Core/Resources/views/js/plugins.js?v='
                . filemtime(PONGHO_PATH . '/Application/Core/Resources/views/js/plugins.js')
            ))
            ->addJavaScript(pongho_url(
                '/Application/Core/Resources/views/js/scripts.js?v='
                . filemtime(PONGHO_PATH . '/Application/Core/Resources/views/js/scripts.js')
            ));

        $this->getHelper()->getHeaderView()
            ->assignVar('body_id', 'new-layout');

        return null;
    }

    /**
     * Controlla la validità dell'url che si vuole impostare per l'account
     *
     * @param $url
     * @return \Pongho\Http\JsonResponse|string
     */
    protected function checkUrl($url)
    {
        $translator = $this->getHelper()->getTranslator();

        // Controllo che non inizi con un numero (serebbe meglio un'espressione regolare)
        if (is_numeric(substr((string) $url, 0, 1))) {
            return $this->getHelper()->displayJsonError($translator->trans('The account url must not start with a number'));
        }

        $url = Inflector::urlize($url);

        /** @var AccountManager $account_manager */
        $account_manager = $this->getContainer()->get('account_manager');

        // Controllo se l'url è già stato usato da altri utenti
        if ($account_manager->findByUrl($url)) {
            return $this->getHelper()->displayJsonError($translator->trans('The account url is already used by another user'));
        }

        return $url;
    }

    /**
     * Azione per cambiare l'url
     *
     * @return \Pongho\Http\JsonResponse
     */
    public function changeurlAction()
    {
        $translator = $this->getHelper()->getTranslator();

        /** @var AccountManager $account_manager */
        $account_manager = $this->getContainer()->get('account_manager');
        $account = $account_manager->findById($this->getParameter('id'));

        if ($this->getRequest()->getMethod() === 'POST') {
            $url = trim((string) $this->getRequest()->post->get('url'));

            // Controllo che mi sia stato passato un url e non un testo vuoto
            if (!$url) {
                return $this->getHelper()->displayJsonError($translator->trans('The account url is not defined'));
            }

            if (($url = $this->checkUrl($url)) instanceof JsonResponse) {
                return $url;
            }

            $message = $translator->trans('The url has been verified, confirm to save');

            if ($this->getRequest()->post->get('action') === 'confirm') {
                $account->old_url = $account->url;
                $account->url = $url;
                $account_manager->save($account);

                $message = $translator->trans('The url has been modified');
            }

            return $this->getHelper()->displayJsonMessage($message, false, ['url' => $url]);
        }

        $view = new View(PONGHO_PATH . '/Application/Core/Resources/views/account_url_edit.php');
        $view->assignVars([
            'translator'  => $translator,
            'form_action' => $this->url('/' . $this->getParameter('path') . '/changeurl/' . $account->id . '/'),
            'account'     => $account,
            'base_url'    => $this->getAccountBaseUrl(),
        ]);

        return new Response($view->render());
    }

    /**
     * @return string
     */
    protected function getAccountBaseUrl()
    {
        /** @var \Pongho\Routing\RoutesCollection $routes */
        $routes = $this->getContainer()->get('routes_collection');

        /** @var \Application\Core\Routes\UserViewRoute $route */
        $route = $routes->get('user-view');

        return absolute_url($route->getPath()) . '/';
    }

    /**
     * {@inheritdoc}
     */
    protected function isExportEnabled()
    {
        return true;
    }
}
