<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Admin\Form\PanelFieldset;
use Application\Core\Form\Field\PermitsChecklistViewClosure;
use Application\Core\Model\Permit;
use Application\Core\Model\Role;
use Pongho\Form\Field\ChecklistField;
use Pongho\Form\Field\TextareaField;
use Pongho\Form\Field\TextField;
use Pongho\Form\Fieldset;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Template\ViewClosure;

class RolesController extends CrudFormController
{
    /**
     * @var int
     */
    private $rowCounter = 1;

    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'roles';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.roles.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Role::class;
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getHelper()->getTranslator()->trans('Roles list');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getHelper()->getTranslator()->trans('Add role');
    }

    /**
     * {@inheritdoc}
     */
    protected function hasAddButton()
    {
        return $this->getHelper()->userHasPermit('users.handle_roles');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return $this->getHelper()->getTranslator()->trans('Add the first role');
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelOptions()
    {
        return [
            'order' => 'position ASC, id ASC',
        ];
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        $translator = $this->getHelper()->getTranslator();

        return [
            [
                'name'   => 'order',
                'label'  => $translator->trans('Order'),
                'class'  => 'order',
                'render' => function ($row) {
                    return $row->order_actions;
                },
            ],
            [
                'name'      => 'name',
                'label'     => $translator->trans('Role'),
                'class'     => 'large',
                'orderable' => true,
                'render'    => function ($row) {
                    return "<strong>{$row->name}</strong><div class=\"actions\"><span>{$row->actions}</span></div>";
                },
            ],
            [
                'name'  => 'description',
                'label' => $translator->trans('Description'),
                'class' => 'main',
            ],
            [
                'name'      => 'id',
                'label'     => 'ID',
                'class'     => 'id',
                'orderable' => true,
            ],
        ];
    }

    /**
     * {@inheritdoc}
     *
     * @param \Application\Core\Model\Role $row
     */
    public function parseArchiveRow($row)
    {
        $translator = $this->getHelper()->getTranslator();
        $path = $this->getParameter('path');

        $actions = [
            [
                $translator->trans('Edit'),
                'href'  => $this->url("/{$path}/edit/{$row->getId()}/"),
                'class' => 'edit',
            ],
        ];

        if ($row->isDeletable()) {
            $actions[] = [
                $translator->trans('delete'),
                'href'  => $this->url("/{$path}/delete/{$row->getId()}/"),
                'class' => 'delete',
            ];
        }

        $row->actions = $this->parseActions($actions);

        // Move Up & Down
        $orderActions = [];

        // Up
        if ($this->rowCounter !== 1) {
            $orderActions[] = [
                $translator->trans('Up'),
                'href'  => $this->url("/{$path}/moveup/{$row->getId()}/"),
                'title' => $translator->trans('Move up'),
                'class' => 'move-up',
            ];
        }

        // Down
        if ($this->rowCounter !== $this->getArchiveManager()->getTotalRecords()) {
            $orderActions[] = [
                $translator->trans('Down'),
                'href'  => $this->url("/{$path}/movedown/{$row->getId()}/"),
                'title' => $translator->trans('Move down'),
                'class' => 'move-down',
            ];
        }

        $this->rowCounter++;
        $row->order_actions = $this->parseActions($orderActions);
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        return $this->getHelper()->filter(
            $this,
            $this->getEventNamespace('filter_form_options'),
            $this->getDefaultFormsOptions(),
            [
                'subject' => $this->getSubject(),
            ]
        );
    }

    /**
     * Restituisce le opzioni di base della form.
     *
     * @return \Application\Admin\Form\FormConfig
     */
    protected function getDefaultFormsOptions()
    {
        $translator = $this->getHelper()->getTranslator();
        $subject = $this->getSubject();

        // Impostazioni principali
        $config = new FormConfig('role', $subject, $this->getHelper()->getLocalization());

        // Struttura
        $config->addBaseStructure(
            $this->getActiveTab(),
            $this->url(str_replace('/pongho/', '/', $this->getRequest()->getRelativeUri()))
        );

        $config->addField(
            'content/main',
            [
                'class' => PanelFieldset::class,
                'label' => $translator->trans('Main'),
            ]
        );

        $config->addField(
            'content/main/role',
            [
                'class'      => Fieldset::class,
                'attributes' => ['class' => 'panel'],
                'settings'   => [
                    'hide_label' => true,
                ],
            ]
        );

        $config->addFields(
            'content/main/role',
            [
                'name'        => [
                    'class'      => TextField::class,
                    'attributes' => [
                        'class'    => 'input_text',
                        'required' => 'required',
                    ],
                    'label'      => $translator->trans('Name'),
                ],
                'description' => [
                    'class' => TextareaField::class,
                    'label' => $translator->trans('Description'),
                ],
            ]
        );

        if ($this->action !== 'add') {
            // Permessi associati
            $config->addField(
                'content/permits',
                [
                    'class' => PanelFieldset::class,
                    'label' => $translator->trans('Permits'),
                ]
            );

            $permits_options = [
                'conditions' => ['is_enabled = ?', true],
                'order'      => '`key`',
            ];

            $lang = $this->getHelper()->getLocalization();

            $permits = [];

            /** @var Permit $permit */
            foreach (Permit::all($permits_options) as $permit) {
                $permit->name = '<strong>' . $lang->get('permit_' . $permit->key) . '</strong>';
                $permits[$permit->id] = $permit;
            }

            $config->addField(
                'content/permits/permits',
                [
                    'class'    => ChecklistField::class,
                    'settings' => [
                        'permits'      => $permits,
                        'localization' => $lang,
                        'view'         => new ViewClosure(
                            new PermitsChecklistViewClosure($lang, $permits)
                        ),
                    ],
                ]
            );
        }

        return $config;
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        if ($model->isNewRecord()) {
            return $this->getHelper()->getTranslator()->trans('Add role');
        }

        return $this->getHelper()->getTranslator()->trans('Edit the role “%title%”', ['%title%' => $model->name]);
    }

    /**
     * @return ModelSubject
     */
    protected function getSubject()
    {
        $subject = new ModelSubject($this->getModel());

        $subject
            ->attachGetter('permits', $this->getPermitsCallback(...))
            ->attachSetter('permits', $this->setPermitsCallback(...));

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject'), $subject);
    }

    /**
     * @return array
     */
    public function getPermitsCallback(ModelSubject $subject)
    {
        /** @var \Application\Core\Model\Role $model */
        $model = $subject->getModel();

        return $model->getPermitsIds();
    }

    public function setPermitsCallback(ModelSubject $subject, array $value)
    {
        /** @var \Application\Core\Model\Role $model */
        $model = $subject->getModel();

        $model->setPermits($value);
    }

    /**
     * {@inheritdoc}
     */
    public function indexAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::indexAction();
    }

    /**
     * {@inheritdoc}
     */
    public function addAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::addAction();
    }

    /**
     * {@inheritdoc}
     */
    public function editAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::editAction();
    }

    /**
     * {@inheritdoc}
     */
    public function deleteAction()
    {
        if (!$this->getHelper()->userHasPermit('users.handle_roles')) {
            throw new HttpUnauthorizedException();
        }

        return parent::deleteAction();
    }

    /**
     * Azione `moveup`.
     *
     * @return \Pongho\Http\Response
     */
    public function moveupAction()
    {
        /** @var \Application\Core\Model\Manager\RoleManager $manager */
        $manager = $this->getContainer()->get('role_manager');

        return $this->move($manager->findById($this->getParameter('id', null, true)), -15);
    }

    /**
     * Azione `movedown`.
     *
     * @return \Pongho\Http\Response
     */
    public function movedownAction()
    {
        /** @var \Application\Core\Model\Manager\RoleManager $manager */
        $manager = $this->getContainer()->get('role_manager');

        return $this->move($manager->findById($this->getParameter('id', null, true)), 15);
    }

    /**
     * Permette lo spostamento del pagamento.
     *
     * @param int  $direction
     * @return \Pongho\Http\Response
     */
    protected function move(Role $role, $direction)
    {
        $role->position += $direction;
        $role->save();

        /** @var Role $role */
        $position = 10;
        foreach (Role::all(['order' => 'position ASC']) as $role) {
            $role->position = $position;
            $role->save();

            $position += 10;
        }

        return $this->getHelper()->redirectResponse(
            $this->url('/' . $this->getParameter('path') . '/')
        );
    }
}
