<?php

/**
 * Questo file è parte di Pongho 2.0-dev.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Core\Model\Permit;
use Pongho\Form\Field\CheckboxField;
use Pongho\Form\Field\TextField;
use Pongho\Form\Subject\ModelSubject;

class PermitsController extends CrudFormController
{
    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'permits';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.permits.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getHelper()->getTranslator()->trans('Permits list');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getHelper()->getTranslator()->trans('Add a new permit');
    }

    /**
     * {@inheritdoc}
     */
    protected function hasAddButton()
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return $this->getHelper()->getTranslator()->trans('Add the first permit');
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Permit::class;
    }

    /**
     * {@inheritdoc}
     *
     * @param Permit $model
     */
    public function getAddEditTitle($model)
    {
        $translator = $this->getHelper()->getTranslator();

        if ($model->isNewRecord()) {
            return $translator->trans('Add permit');
        }

        return $translator->trans(
            'Edit the permit “%title%”',
            [
                '%title%' => $this->getHelper()->getLocalization()->get('permit_' . $model->key),
            ]
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        $translator = $this->getHelper()->getTranslator();

        $lang = $this->getHelper()->getLocalization();

        return [
            [
                'name'      => 'name',
                'label'     => $translator->trans('Permit'),
                'class'     => 'main',
                'orderable' => true,
                'render'    => function (Permit $row) use ($lang) {
                    // todo: usare translator quando le traduzioni per i nomi dei
                    //       permessi verranno gestite con una tabella apposita
                    $keyTranslation = $lang->get('permit_' . $row->key);

                    return "<strong>{$keyTranslation}</strong><div class=\"actions\"><span>{$row->actions}</span></div>";
                },
            ],
            [
                'name'      => 'key',
                'label'     => $translator->trans('Key'),
                'class'     => 'large',
                'orderable' => true,
            ],
            [
                'name'      => 'is_enabled',
                'label'     => $translator->trans('Is enabled'),
                'class'     => 'large',
                'orderable' => true,
                'render'    => function (Permit $row) use ($translator) {
                    return $row->is_enabled ? $translator->trans('Yes') : $translator->trans('No');
                },
            ],
        ];
    }

    /**
     * @return ModelSubject
     */
    public function getSubject()
    {
        return new ModelSubject($this->getModel());
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        $subject = $this->getSubject();
        $translator = $this->getHelper()->getTranslator();

        $options = new FormConfig('permits', $subject, $this->getHelper()->getLocalization());

        $options->addBaseStructure('main', $this->currentUrl());
        $options->addTab('content/main', $translator->trans('Main'));
        $options->addPanel('content/main/permit-panel');

        $options->addFields('content/main/permit-panel', [
            'key'        => [
                'class' => TextField::class,
                'label' => $translator->trans('Permit key'),
            ],
            'is_enabled' => [
                'class' => CheckboxField::class,
                'label' => $translator->trans('Is enabled'),
            ],
        ]);

        return $this->getHelper()->filter(
            $this,
            $this->getEventNamespace('filter_form_options'),
            $options,
            [
                'subject' => $subject,
            ]
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelFieldsConfig()
    {
        $translator = $this->getHelper()->getTranslator();

        return [
            'key'        => [
                'label' => $translator->trans('Permit key'),
            ],
            'is_enabled' => [
                'label' => $translator->trans('Is enabled'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    protected function getFilterEnumValues()
    {
        $translator = $this->getHelper()->getTranslator();

        return [
            'is_enabled' => [
                '0' => $translator->trans('No'),
                '1' => $translator->trans('Yes'),
            ],
        ];
    }
}
