<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\Exception\TranslationFieldException;
use Application\Admin\Form\FormConfig;
use Application\Admin\Form\Subject\TranslationsSubject;
use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\NotificationTemplateInterface;
use Application\Core\Model\Language;
use Application\Core\Model\Manager\LanguageManager;
use Application\Core\Model\Manager\NotificationTemplateManagerInterface;
use Application\Core\Model\NotificationTemplate;
use Application\Core\Model\NotificationTemplateTranslation;
use Application\Core\Notification\NotificationCollection;
use Application\Core\Notification\NotificationInterface;
use Pongho\Form\Fieldset;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\RedirectResponse;
use Pongho\Template\ViewClosure;
use Pongho\Utilities\Inflector;

class NotificationsTemplatesController extends CrudFormController
{
    /**
     * @var ModelSubject
     */
    protected $subject;

    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'notifications';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.notifications.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getHelper()->getTranslator()->trans('Notification templates');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText(): never
    {
        throw new \BadFunctionCallException('This controller does not support adding new models');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText(): never
    {
        throw new \BadFunctionCallException('This controller does not support adding new models');
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return \Application\Core\Model\NotificationTemplate::class;
    }

    /**
     * @return RedirectResponse
     */
    public function resetAction()
    {
        $id = $this->getParameter('id');

        /** @var NotificationTemplateManagerInterface $manager */
        $manager = $this->getContainer()->get('notification_template_manager');

        /** @var NotificationTemplate $model */
        $model = $manager->findById($id);

        if ($model) {
            if ($model->delete()) { // todo: il delete deve farlo il manager
                return $this->getHelper()->displayJsonMessage(
                    'Ok!',
                    false,
                    [
                        'redirect' => $this->url(
                            '/' . $this->getParameter('path') . '/add/' . $model->getNotificationName() . '/',
                            '&'
                        ),
                    ]
                );
            } else {
                return $this->getHelper()->displayJsonError('Template reset failed');
            }
        }

        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        return $this->getHelper()->getTranslator()->trans(
            'Edit the notification “%title%”',
            ['%title%' => $model->getNotificationName()]
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveManager(?array $config = null)
    {
        $manager = $this->getContainer()->get('notification_template_manager');

        $config = [
            'model_manager_all'   => [[$manager, 'findAll']],
            'model_manager_count' => [[$manager, 'countAll']],
            'model_fields'        => $this->getModelFieldsConfigWrapper(),
            'parse_row'           => $this->parseArchiveRow(...),
            'search_fields'       => $this->getSearchFieldsWrapper(),
            'fast_filters'        => $this->getFastFilters(),
            'form'                => $this->getFiltersForm(...),
            'table'               => [
                'columns'         => $this->convertTableColumns(...),
                'empty_recordset' => $this->getTableEmptyRecordsetText(),
                'show_ckb'        => false,
                'show_add'        => false,
            ],
        ];

        return parent::getArchiveManager($config);
    }

    /**
     * {@inheritdoc}
     */
    protected function getMassActions()
    {
        return [];
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        $translator = $this->getHelper()->getTranslator();

        /** @var NotificationCollection $collection */
        $collection = $this->getContainer()->get('notification_collection');

        return [
            'title'  => [
                'label'  => $translator->trans('Name'),
                'class'  => 'main',
                'render' => function (NotificationTemplate $row) use ($collection) {
                    return sprintf(
                        '<strong>%s</strong><div class="actions"><span>%s</span></div>',
                        $collection->get($row->getNotificationName())->getTitle(),
                        $row->actions
                    );
                },
            ],
            'name'   => [
                'label'  => $translator->trans('Notification'),
                'class'  => 'large',
                'render' => function (NotificationTemplate $row) {
                    return $row->getNotificationName();
                },
            ],
            'custom' => [
                'label'  => $translator->trans('Customized'),
                'class'  => 'ckb',
                'render' => function (NotificationTemplate $row) {
                    return $row->isNewRecord() ? '&nbsp;' : '&#10004;';
                },
            ],
        ];
    }

    /**
     * Aggiunge le azioni alle righe dell'elenco dei modelli
     *
     * @param NotificationTemplate $row
     */
    public function parseArchiveRow($row)
    {
        $path = $this->getParameter('path');
        $translator = $this->getHelper()->getTranslator();

        if ($row->isNewRecord()) {
            $actions = [
                [$translator->trans('Edit'), 'href' => $this->url("/{$path}/add/{$row->notification}/"), 'class' => 'edit'],
            ];
        } else {
            $actions = [
                [$translator->trans('Edit'), 'href' => $this->url("/{$path}/edit/{$row->id}/"), 'class' => 'edit'],
            ];
        }

        $row->actions = $this->parseActions($actions);
    }

    /**
     * @return NotificationTemplateInterface|NotificationTemplate
     */
    protected function getModel()
    {
        /** @var NotificationTemplateManagerInterface $manager */
        $manager = $this->getContainer()->get('notification_template_manager');

        $id = $this->getParameter('id');

        if (is_numeric($id)) {
            if ($model = $manager->findById($id)) {
                return $model;
            }
        } else {
            $name = $this->getNotificationNameFromPath();

            if ($name) {
                /** @var NotificationTemplate $model */
                $model = $manager->findByNotification($name);

                if ($model) {
                    $model->site_id = $this->getHelper()->getSiteId();

                    return $model;
                }
            }
        }

        throw new HttpNotFoundException();
    }

    /**
     * @return string
     */
    protected function getNotificationNameFromPath()
    {
        $path = $this->getRequest()->getPathInfo();
        $segments = explode('/', trim($path, '/'));

        return $segments[3] ?? null;
    }

    /**
     * @return string
     */
    protected function getFormAction()
    {
        $path = $this->getParameter('path');

        // Gestione dell'azione
        if ($this->action === 'add') {
            return $this->url('/' . $path . '/add/' . $this->getNotificationNameFromPath() . '/');
        } else {
            return $this->url('/' . $path . '/edit/' . $this->getModel()->id . '/');
        }
    }

    /**
     * @return ModelSubject
     */
    protected function getSubject()
    {
        if ($this->subject === null) {
            $this->subject = new ModelSubject($this->getModel());

            $this->subject->attachGetter(
                'translations',
                function (ModelSubject $subject) {
                    /** @var NotificationTemplate $template */
                    $template = $subject->getModel();
                    $translations = $template->getTranslations();

                    $subjects = [];
                    foreach ($this->getLanguages() as $language) {
                        if (array_key_exists($language->id, $translations)) {
                            $translation = $translations[$language->id];
                        } else {
                            $translation = new NotificationTemplateTranslation();
                        }

                        $translation_subject = new ModelSubject($translation);

                        $translation_subject->setBeforeSaveCallback(function (ModelSubject $subject) use ($language, $template): void {
                            if ($subject->get('subject_to_user') === '') {
                                throw new TranslationFieldException('The field "%s" is required', 'subject_to_user', $language->id);
                            }

                            if ($subject->get('language_id') === null) {
                                $subject->set('language_id', $language->id);
                            }

                            $subject->set('template_id', $template->getId());
                        });

                        $subjects[$language->id] = $translation_subject;
                    }

                    return new TranslationsSubject($subjects);
                }
            );
        }

        return $this->subject;
    }

    protected $languages;

    /**
     * @return array
     */
    protected function getLanguages()
    {
        if ($this->languages === null) {
            $this->languages = [];

            /** @var LanguageManager $manager */
            $manager = $this->getContainer()->get('language_manager');

            /** @var Language $language */
            foreach ($manager->findAllBySite($this->getSiteId()) as $language) {
                $this->languages[$language->getId()] = $language;
            }
        }

        return $this->languages;
    }

    /**
     * Callback "before_set_template".
     */
    protected function beforeSetTemplateCallback()
    {
        $this->getHelper()->getHeaderView()
            ->assignVar('body_id', 'new-layout');

        $reset_confirm_message = $this->getHelper()
            ->getTranslator()->trans('Do you want to reset the template to its initial status?');

        // fixme con più di un editor non funziona correttamente
        $this->getHelper()
            ->loadEditorWysiwyg([], $this)
            ->addJavaScriptInline(
                <<<JS
$(document).ready(function () {
    var onCreatePanel = function (el) {
        $("dt.newsletter-code").click(function () {
            $('.tabsPanel:visible .tinymce-newsletter').tinymce().execCommand('mceInsertContent', false, $(this).text());
        });

        $('#button-notifications-reset').click(function () {
            if (confirm('{$reset_confirm_message}') == true) {
                var btn = $(this);

                btn.addClass('busy');

                $.ajax({
                    'url': btn.data('url'),
                    'method': 'POST',
                    'success': function (response) {
                        if (response.error) {
                            btn.removeClass('busy');
                            alert(response.message);
                        }

                        if (response.redirect) {
                            window.location.replace(response.redirect);
                        }
                    }
                });
            }
        });
    }, tabsPanel = $('.form-wrapper');

    if (Pongho.action !== 'add') {
        tabsPanel.on('after-update', function () {
            onCreatePanel($(this));
        });
    }

    onCreatePanel(tabsPanel);
});
JS
            );
    }

    /**
     * Callback per la vista della sidebar
     *
     * @return string
     */
    public function sidebarVarsViewCallback(Fieldset $field)
    {
        $translator = $this->getHelper()->getTranslator();

        /**
         * @var NotificationCollection        $collection
         * @var NotificationTemplateInterface $model
         */
        $collection = $this->getContainer()->get('notification_collection');
        $model = $field->getSubject()->getModel();
        $notification = $collection->get($model->getNotificationName());

        $code = '';
        foreach ($notification->getVars() as $var) {
            $code .= sprintf(
                '<dt class="newsletter-code">%%%s%%</dt>',
                $var
            );
        }

        return sprintf(
            '%s<h3>%s</h3><p>%s</p><dl class="email_vars">%s</dl>%s',
            $field->getStartCode(),
            $translator->trans('Codes'),
            $translator->trans('You can use these codes to insert some dynamic data in the message'),
            $code,
            $field->getEndCode()
        );
    }

    /**
     * Callback per la vista del messaggio
     *
     * @return string
     */
    public function messageForAdminViewCallback(Fieldset $field)
    {
        return <<<HTML
{$field->getStartCode()}
    <div class="control-group">{$field->getGenerator()->render($field->getField('subject_to_admin'))}</div>
    <div class="control-group">{$field->getField('content_to_admin')->getTag()}</div>
{$field->getEndCode()}
HTML;
    }

    /**
     * Callback per la vista del messaggio
     *
     * @return string
     */
    public function messageforUserViewCallback(Fieldset $field)
    {
        return <<<HTML
{$field->getStartCode()}
    <div class="control-group">{$field->getGenerator()->render($field->getField('subject_to_user'))}</div>
    <div class="control-group">{$field->getField('content_to_user')->getTag()}</div>
{$field->getEndCode()}
HTML;
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        $translator = $this->getHelper()->getTranslator();

        $config = new FormConfig('notifications', $this->getSubject(), $translator);

        /** @var NotificationTemplate $model */
        $model = $this->getSubject()->getModel();

        $config->addSidebarStructure();

        // todo: completare la funzionalità per l'invio dell'anteprima
        // Pulsante per l'anteprima dell'email (invia tutte le email all'indirizzo dell'amministratore del sito)
        // $config->addField(
        //     'actions/preview',
        //     array(
        //         'class' => 'Pongho\\Form\\Field\\ButtonField',
        //         'label' => $translator->trans('Send preview email'),
        //     )
        // );

        // Pulsante per il ripristino del template, lo metto sulla destra per evitare di cliccarlo per errore
        if (!$model->isNewRecord()) {
            $config->addField(
                'actions/reset',
                [
                    'class'      => \Pongho\Form\Field\ButtonField::class,
                    'label'      => $translator->trans('Reset the template'),
                    'attributes' => [
                        'style'    => 'float: right;',
                        'data-url' => $this->url('/' . $this->getParameter('path') . '/reset/' . $model->getId() . '/'),
                    ],
                ]
            );
        }

        // Pannello di configurazione
        $config->addField(
            'content/wrapper/container/config',
            [
                'class' => \Application\Admin\Form\PanelsManagerFieldset::class,
            ]
        );

        // Sidebar
        $config->addPanel(
            'content/wrapper/sidebar/vars',
            null,
            false,
            [
                'view' => new ViewClosure($this->sidebarVarsViewCallback(...)),
            ]
        );

        /**
         * @var NotificationCollection        $collection
         * @var NotificationTemplateInterface $model
         */
        $collection = $this->getContainer()->get('notification_collection');
        $notification = $collection->get($model->getNotificationName());

        // Campi di configurazione del template
        $config = $this->addAdminFieldsToConfig($config, $notification);

        return $this->addUserFieldsToConfig($config, $notification);
    }

    /**
     * @return FormConfig
     */
    protected function addAdminFieldsToConfig(FormConfig $config, NotificationInterface $notification)
    {
        if ($notification->getOptions()->canSendToAdmin()) {
            $translator = $this->getHelper()->getTranslator();

            $config->addTab('content/wrapper/container/config/admin', $translator->trans('Notify to admin'));

            $config->addPanel('content/wrapper/container/config/admin/addresses');
            $config->addPanel(
                'content/wrapper/container/config/admin/message',
                null,
                false,
                ['view' => new ViewClosure($this->messageForAdminViewCallback(...))]
            );

            $attributes = [];
            if (!$notification->getOptions()->canChangeSenderToAdmin()) {
                $attributes['readonly'] = 'readonly';
            }

            $config->addField(
                'content/wrapper/container/config/admin/addresses/sender_to_admin',
                [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Sender'),
                    'attributes' => $attributes,
                ]
            );

            $attributes = [];
            if (!$notification->getOptions()->canChangeRecipientToAdmin()) {
                $attributes['readonly'] = 'readonly';
            }

            $config->addFieldAfter(
                'content/wrapper/container/config/admin/addresses/sender_to_admin',
                'recipient_to_admin',
                [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Recipient'),
                    'attributes' => $attributes,
                ]
            );

            if ($notification->getOptions()->canReplyToAdmin()) {
                $attributes = [];
                if (!$notification->getOptions()->canChangeReplyToAdmin()) {
                    $attributes['readonly'] = 'readonly';
                }

                $config->addFieldAfter(
                    'content/wrapper/container/config/admin/addresses/sender_to_admin',
                    'reply_to_admin',
                    [
                        'class'      => \Pongho\Form\Field\TextField::class,
                        'label'      => $translator->trans('Reply to'),
                        'attributes' => $attributes,
                    ]
                );
            }

            $config->addFields(
                'content/wrapper/container/config/admin/message',
                [
                    'subject_to_admin' => [
                        'class' => \Pongho\Form\Field\TextField::class,
                        'label' => $translator->trans('Subject'),
                    ],
                    'content_to_admin' => [
                        'class'      => \Pongho\Form\Field\TextareaField::class,
                        'attributes' => ['class' => 'tinymce-newsletter', 'rows' => 25],
                        'label'      => $translator->trans('Message'),
                    ],
                ]
            );
        }

        return $config;
    }

    /**
     * @return FormConfig
     */
    protected function addUserFieldsToConfig(FormConfig $config, NotificationInterface $notification)
    {
        if ($notification->getOptions()->canSendToUser()) {
            $translator = $this->getHelper()->getTranslator();

            $config->addTab('content/wrapper/container/config/user', $translator->trans('Notify to user'));

            $languages = $this->getLanguages();

            $config->addPanel('content/wrapper/container/config/user/addresses');

            // Traduzioni
            $config->addField(
                'content/wrapper/container/config/user/translations',
                [
                    'class'    => \Application\Admin\Form\TranslationsManagerFieldset::class,
                    'subject'  => $this->getSubject(),
                    'settings' => [
                        'languages'            => $languages,
                        'tab_active'           => $this->getActiveTranslationTab(),
                        'tab_prefix'           => 'language',
                        'tab_url'              => $this->currentUrl(),
                        'priority'             => 50, // salvo le traduzioni per ultime
                        'enabled_tab_callback' => function (LanguageInterface $language) use ($languages) {
                            return true; // Uasando il manager mi arrivano solo le lingue attive per il sito
                        },
                    ],
                ]
            );
            $config->addPanel(
                'content/wrapper/container/config/user/translations/message',
                null,
                false,
                ['view' => new ViewClosure($this->messageforUserViewCallback(...))]
            );

            $attributes = [];
            if (!$notification->getOptions()->canChangeSenderToUser()) {
                $attributes['readonly'] = 'readonly';
            }

            $config->addField(
                'content/wrapper/container/config/user/addresses/sender_to_user',
                [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Sender'),
                    'attributes' => $attributes,
                ]
            );

            $attributes = [];
            if (!$notification->getOptions()->canChangeRecipientToUser()) {
                $attributes['readonly'] = 'readonly';
            }

            $config->addFieldAfter(
                'content/wrapper/container/config/user/addresses/sender_to_user',
                'recipient_to_user',
                [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Recipient'),
                    'attributes' => $attributes,
                ]
            );

            if ($notification->getOptions()->canReplyToUser()) {
                $attributes = [];
                if (!$notification->getOptions()->canChangeReplyToUser()) {
                    $attributes['readonly'] = 'readonly';
                }

                $config->addFieldAfter(
                    'content/wrapper/container/config/user/addresses/sender_to_user',
                    'reply_to_user',
                    [
                        'class'      => \Pongho\Form\Field\TextField::class,
                        'label'      => $translator->trans('Reply to'),
                        'attributes' => $attributes,
                    ]
                );
            }

            $config->addFields(
                'content/wrapper/container/config/user/translations/message',
                [
                    'subject_to_user' => [
                        'class' => \Pongho\Form\Field\TextField::class,
                        'label' => $translator->trans('Subject'),
                    ],
                    'content_to_user' => [
                        'class'      => \Pongho\Form\Field\TextareaField::class,
                        'label'      => $translator->trans('Message'),
                        'attributes' => ['class' => 'tinymce-newsletter', 'rows' => 25],
                    ],
                ]
            );
        }

        return $config;
    }

    /**
     * Restituisce la chiave della tab che deve essere attivata
     *
     * @return string
     */
    protected function getActiveTab()
    {
        return $this->getRequest()->query->get(
            'tabs[tab]',
            $this->getRequest()->post->get('tabs[tab]', 'admin', true),
            true
        );
    }

    /**
     * Restituisce la chiave della traduzione che deve essere attivata
     *
     * @return string
     */
    protected function getActiveTranslationTab()
    {
        /** @var Language $language */
        $language = Language::first();

        return $this->getRequest()->query->get(
            'tabs[language]',
            $this->getRequest()->post->get('tabs[language]', Inflector::selectorize($language->culture), true),
            true
        );
    }
}
