<?php

namespace Application\Core\Controller\Admin\LinkField;

use Application\Cms\Model\Taxonomy;
use Application\Cms\Model\Term;
use Application\Core\Model\LanguageModule;
use Application\Core\Model\LanguageSite;
use Application\Core\Model\LinkType;
use Application\Core\Model\Module;
use Application\Core\Model\ModuleSite;
use Pongho\Core\Kernel;

class TermHandler extends \Application\Admin\Form\LinkField\BaseHandler
{
    private $taxonomies = [];

    private $terms = [];

    /**
     * {@inheritdoc}
     */
    public function getClass($type = '')
    {
        return static::class . '::term-' . $type;
    }

    /**
     * {@inheritdoc}
     */
    public function addTypesOptions($language_id, array $options)
    {
        foreach ($this->getTaxonomies($language_id) as $taxonomy) {
            $key = 'term-' . $taxonomy->node_type . '-' . $taxonomy->name;

            if ($language_id === null) {
                $options[$taxonomy->language_id][$this->getClass(
                    $taxonomy->node_type . '-' . $taxonomy->name . '_' . $taxonomy->language_id
                )] = $key;
            } else {
                $options[$this->getClass($taxonomy->node_type . '-' . $taxonomy->name)] = $key;
            }
        }

        return $options;
    }

    /**
     * {@inheritdoc}
     */
    public function render($name, $language_id, ?LinkType $link_type = null)
    {
        $html = Kernel::instance()->getContainer()->get('template_html');

        $blocks = [];
        foreach ($this->getTaxonomies($language_id) as $taxonomy) {
            $key = 'term-' . $taxonomy->node_type . '-' . $taxonomy->name . ($language_id === null ? '_' . $taxonomy->language_id : '');
            $field_name = $name . '[' . $key . ']';
            $language = $language_id ?? $taxonomy->language_id;

            $options = $html->selectHierarchy(
                $this->getTerms($taxonomy, $language),
                ($link_type instanceof LinkType ? $link_type->model_id : null)
            );

            $blocks[$key] = '<select name="' . $field_name . '">' . $options . '</select>';
        }

        return $blocks;
    }

    public function renderSpan($name, $language_id, ?LinkType $link_type = null)
    {
        $blocks = [];
        foreach ($this->getTaxonomies($language_id) as $taxonomy) {
            $language = ($language_id ?? $taxonomy->language_id);

            $key = 'term-' . $taxonomy->node_type . '-' . $taxonomy->name . '_' . $language;
            $field_name = $name . '[' . $key . ']';

            $blocks[$key] = '<span data-name="' . $field_name . '" data-value="' . ($link_type instanceof LinkType ? $link_type->model_id : '') . '" data-language="' . $language . '"></span>';
        }

        return $blocks;
    }

    public function getSelectOptions($language_id, array $query_options = [])
    {
        $html = Kernel::instance()->getContainer()->get('template_html');

        $blocks = [];
        foreach ($this->getTaxonomies($language_id) as $taxonomy) {
            $language = ($language_id ?? $taxonomy->language_id);

            $key = 'term-' . $taxonomy->node_type . '-' . $taxonomy->name . '_' . $language;

            $blocks[$key] = $html->selectHierarchy($this->getTerms($taxonomy, $language));
        }

        return $blocks;
    }

    private function getTaxonomies($language_id)
    {
        if (!array_key_exists($language_id, $this->taxonomies)) {
            if ($language_id === null) {
                $options = [
                    'select'     => 'lm.*, ms.*, ls.*, m.node_type, `from`.*',
                    'joins'      => 'INNER JOIN ' . ModuleSite::tableName() . ' AS ms ON ms.id = `from`.module_site_id'
                                 . ' INNER JOIN ' . Module::tableName() . ' AS m ON m.id = ms.module_id'
                                 . ' INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.site_id = ms.site_id'
                                 . ' LEFT JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.module_id = m.id AND lm.language_site_id = ls.id',
                    'conditions' => [
                        'ms.site_id = :site AND ms.is_enabled = :enabled AND lm.is_enabled = :enabled',
                        'site'    => $this->site_id,
                        'enabled' => true,
                    ],
                ];

                $this->taxonomies[null] = Taxonomy::all($options);
            } else {
                $this->taxonomies[$language_id] = Taxonomy::findAllBySiteAndlanguage($this->site_id, $language_id);
            }
        }

        return $this->taxonomies[$language_id];
    }

    private function getTerms(Taxonomy $taxonomy, $language_id)
    {
        $tax_key = $taxonomy->node_type . '-' . $taxonomy->name;
        if (!isset($this->terms[$tax_key][$language_id])) {
            $terms = [];
            foreach ($taxonomy->getTerms($language_id) as $term) {
                $terms[$term->parent_id][$term->id] = $term->title;
            }

            $this->terms[$tax_key][$language_id] = $terms;
        }

        return $this->terms[$tax_key][$language_id];
    }

    /**
     * {@inheritdoc}
     */
    public static function parseAttributes(array $attributes)
    {
        $key = $attributes['key'] ?? $attributes['link_type'];

        if (array_key_exists($key, $attributes)) {
            $attributes['model_id'] = $attributes[$key];

            $term = Term::find($attributes['model_id'], $attributes['language_id']);

            $attributes['model_path'] = $term->permalink;
        }

        return $attributes;
    }
}
