<?php

namespace Application\Core\Controller\Admin\LinkField;

use Application\Admin\Form\LinkField\LinkField;
use Application\Core\Localization;
use Application\Core\Model\LinkType;
use Pongho\Template\Html;

class SeoLinkField extends LinkField
{
    /**
     * Restituisce l'elenco delle opzioni per la select iniziale.
     */
    public function getTypesOptions($language_id)
    {
        $options = [];
        foreach ($this->handlers as $handler) {
            $options = $handler->addTypesOptions($language_id, $options);
        }

        $result = [];
        foreach ($options as $value => &$label) {
            if (($language_id === null) && is_numeric($value) && is_array($label)) {
                $items = [];
                foreach ($label as $key => $item) {
                    $items[$key] = [
                        $this->lang->get('link_type_' . str_replace('-', '_', $item)) . ' (' . strtoupper((string) $this->languages[$value]->iso) . ')',
                        'data-value' => $item . '_' . $value,
                    ];
                }

                $label = [
                    'optgroup' => $items,
                ];

                $value = $this->languages[$value]->name;
            } else {
                $label = [
                    $this->lang->get('link_type_' . str_replace('-', '_', $label)),
                    'data-value' => $label,
                ];
            }

            $result[$value] = $label;
        }

        return $result;
    }

    /**
     * Restituisce il codice HTML del campo.
     */
    public function render($name, $language_id, ?LinkType $link_type = null)
    {
        $link_type_handler = null;

        if ($link_type instanceof LinkType) {
            $link_type_handler = $link_type->handler . '::' . $link_type->link_type;

            if ($language_id === null && $link_type->language_id) {
                $link_type_handler .= '_' . $link_type->language_id;
            }
        }

        $select = $this->select($language_id, $link_type_handler);

        $blocks = [
            '<div><select name="' . $name . '[link_type]" class="link-type-select">' . $select . '</select></div>',
        ];

        foreach ($this->handlers as $handler) {
            foreach ($handler->renderSpan($name, $language_id, $link_type) as $block_name => $block_code) {
                $language = '';
                if ($language_id === null && $link_type && $link_type->language_id) {
                    $language = '_' . $link_type->language_id;
                }

                $display = ($link_type instanceof LinkType && (($link_type->link_type . $language) == $block_name)) ? 'block' : 'none';

                $blocks[] = '<div class="link-type link-type-' . $block_name . '" style="display:' . $display . ';">' . $block_code . '</div>';
            }
        }

        return implode('', $blocks);
    }

    /**
     *
     */
    public function setHandlers($field)
    {
        $handlers = [
            'url'      => [
                'nolink' => \Application\Admin\Form\LinkField\NolinkHandler::class,
                'link'   => \Application\Admin\Form\LinkField\LinkHandler::class,
                'module' => \Application\Core\Controller\Admin\LinkField\ModuleHandler::class,
                'node'   => \Application\Core\Controller\Admin\LinkField\NodeHandler::class,
                'term'   => \Application\Core\Controller\Admin\LinkField\TermHandler::class,
            ],
            'redirect' => [
                'nolink' => \Application\Admin\Form\LinkField\NolinkHandler::class,
                'link'   => \Application\Admin\Form\LinkField\LinkHandler::class,
                'module' => \Application\Core\Controller\Admin\LinkField\ModuleHandler::class,
                'node'   => \Application\Core\Controller\Admin\LinkField\NodeHandler::class,
                'term'   => \Application\Core\Controller\Admin\LinkField\TermHandler::class,
            ],
        ];

        if (array_key_exists($field, $handlers)) {
            foreach ($handlers[$field] as $name => $handler) {
                $this->addHandler($name, $handler);
            }
        }
    }
}
