<?php

namespace Application\Core\Controller\Admin\LinkField;

use Application\Cms\Model\Node;
use Application\Cms\Model\NodeType;
use Application\Core\Model\LanguageModule;
use Application\Core\Model\LanguageSite;
use Application\Core\Model\LinkType;
use Application\Core\Model\Module;
use Pongho\Core\Kernel;

class NodeHandler extends \Application\Admin\Form\LinkField\BaseHandler
{
    private $nodes_types = [];

    private $nodes = [];

    public function getClass($type = '')
    {
        return static::class . '::node-' . $type;
    }

    public function addTypesOptions($language_id, array $options)
    {
        foreach ($this->getNodesTypes($language_id) as $node_type) {
            if (!$node_type->is_menu_enabled) {
                continue;
            }

            if ($language_id === null) {
                $options[$node_type->language_id][$this->getClass($node_type->node_type . '_' . $node_type->language_id)] = 'node-' . $node_type->node_type;
            } else {
                $options[$this->getClass($node_type->node_type)] = 'node-' . $node_type->node_type;
            }
        }

        return $options;
    }

    public function render($name, $language_id, ?LinkType $link_type = null)
    {
        $blocks = [];
        $html = Kernel::instance()->getContainer()->get('template_html');

        foreach ($this->getNodesTypes($language_id) as $node_type) {
            if (!$node_type->is_menu_enabled) {
                continue;
            }

            $key = 'node-' . $node_type->node_type . ($language_id === null ? '_' . $node_type->language_id : '');
            $language = $language_id ?? $node_type->language_id;
            $blocks[$key] = '<select name="' . $name . '[' . $key . ']">' . $html->selectHierarchy($this->getNodes($node_type, $language), ($link_type instanceof LinkType ? $link_type->model_id : null)) . '</select>';
        }

        return $blocks;
    }

    public function renderSpan($name, $language_id, ?LinkType $link_type = null)
    {
        $blocks = [];
        foreach ($this->getNodesTypes($language_id) as $node_type) {
            if (!$node_type->is_menu_enabled) {
                continue;
            }

            $language = $language_id ?? $node_type->language_id;
            $key = 'node-' . $node_type->node_type . '_' . $language;
            $blocks[$key] = '<span data-name="' . $name . '[' . $key . ']" data-value="' . ($link_type instanceof LinkType ? $link_type->model_id : null) . '" data-language="' . $language . '"></span>';
        }

        return $blocks;
    }

    public function getSelectOptions($language_id, array $query_options = [])
    {
        $html = Kernel::instance()->getContainer()->get('template_html');

        $blocks = [];
        foreach ($this->getNodesTypes($language_id) as $node_type) {
            if (!$node_type->is_menu_enabled) {
                continue;
            }

            $language = $language_id ?? $node_type->language_id;
            $key = 'node-' . $node_type->node_type . '_' . $language;
            $blocks[$key] = $html->selectHierarchy($this->getNodes($node_type, $language));
        }

        return $blocks;
    }

    private function getNodesTypes($language_id = null)
    {
        if (!array_key_exists($language_id, $this->nodes_types)) {
            if ($language_id === null) {
                $options = [
                    'select'     => 'm.*, ls.*, lm.*, `from`.*',
                    'joins'      => 'INNER JOIN ' . Module::tableName() . ' AS m ON m.id = `from`.module_id'
                                 . ' INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.site_id = `from`.site_id'
                                 . ' INNER JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.module_id = `from`.module_id AND lm.language_site_id = ls.id',
                    'conditions' => [
                        "`from`.site_id = :site AND m.node_type <> '' AND `from`.is_enabled = :enabled AND lm.is_enabled = :enabled",
                        'site'    => $this->site_id,
                        'enabled' => true,
                    ],
                ];

                $this->nodes_types[null] = NodeType::all($options);
            } else {
                $this->nodes_types[$language_id] = NodeType::findAll($this->site_id, $language_id);
            }
        }

        return $this->nodes_types[$language_id];
    }

    private function getNodes(NodeType $type, $language_id)
    {
        if (!isset($this->nodes[$type->id][$language_id])) {
            $options = [
                'conditions' => [
                    "`from`.module_site_id = :module_site AND `from`.status IN ('publish', 'home') AND t.is_enabled = :enabled",
                    'module_site' => $type->id,
                    'enabled'     => true,
                ],
            ];

            $nodes = [];
            foreach (Node::all($options, $language_id) as $node) {
                $nodes[$node->parent_id][$node->id] = $node->title;
            }

            $this->nodes[$type->id][$language_id] = $nodes;
        }

        return $this->nodes[$type->id][$language_id];
    }

    public static function parseAttributes(array $attributes)
    {
        $key = $attributes['key'] ?? $attributes['link_type'];

        if (array_key_exists($key, $attributes)) {
            $attributes['model_id'] = $attributes[$key];

            $node = Node::find($attributes['model_id'], $attributes['language_id']);

            $attributes['model_path'] = $node->permalink;
        }

        return $attributes;
    }
}
