<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudController;
use Application\Admin\Form\TextareaField;
use Application\Core\Model\Account;
use Application\Core\Model\Group;
use Application\Core\Model\GroupUser;
use Pongho\Http\Exception\HttpException;

/**
 * @todo Passare al CrudFormController
 * @todo Rivedere completamente l’interfaccia per gestire l’associazione con gli utenti
 */
class GroupsController extends CrudController
{
    /**
     * {@inheritdoc}
     */
    public function addAction()
    {
        ini_set('memory_limit', '2048M');

        return parent::addAction();
    }

    /**
     * {@inheritdoc}
     */
    public function editAction()
    {
        ini_set('memory_limit', '2048M');

        return parent::editAction();
    }

    /**
     * {@inheritdoc}
     */
    public function displayArchive($page = 1)
    {
        $translator = $this->getHelper()->getTranslator();
        $path = $this->getParameter('path');

        /** @var Group[] $groups */
        $groups = Group::all([
            'order' => 'position ASC, name ASC',
        ]);

        $rowCounter = 1;
        $groupsTotal = count($groups);
        foreach ($groups as $group) {
            $actions = [
                ['edit', 'href' => $this->url("/{$path}/edit/{$group->getId()}/"), 'class' => 'edit'],
                ['delete', 'href' => $this->url("/{$path}/delete/{$group->getId()}/"), 'class' => 'delete'],
            ];

            $group->actions = $this->parseActions($actions);

            // Move Up & Down
            $orderActions = [];

            // Up
            if ($rowCounter !== 1) {
                $orderActions[] = [
                    $translator->trans('Up'),
                    'href'  => $this->url("/{$path}/moveup/{$group->getId()}/"),
                    'title' => $translator->trans('Move up'),
                    'class' => 'move-up',
                ];
            }

            // Down
            if ($rowCounter !== $groupsTotal) {
                $orderActions[] = [
                    $translator->trans('Down'),
                    'href'  => $this->url("/{$path}/movedown/{$group->getId()}/"),
                    'title' => $translator->trans('Move down'),
                    'class' => 'move-down',
                ];
            }

            $rowCounter++;
            $group->order_actions = $this->parseActions($orderActions);
        }

        $this->getHelper()->addCss(pongho_url(
            '/Application/Admin/Resources/views/css/style-legacy.css?v='
            . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/css/style-legacy.css')
        ));

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/groups_list.php')
            ->assignVars([
                'rows'    => $groups,
                'add_url' => $this->url('/groups/add/'),
            ]);
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Group::class;
    }

    /**
     * {@inheritdoc}
     *
     * @param Group $model
     */
    public function getTitle($model)
    {
        if ($this->getAction() === 'add') {
            return $this->getHelper()->getTranslator()->trans('Add group');
        }

        return sprintf($this->getHelper()->getTranslator()->trans('Edit group'), $model->getName());
    }

    /**
     * {@inheritdoc}
     *
     * @param Group $model
     */
    public function getFields($model)
    {
        return $this->getHelper()->filter(
            $this,
            'admin.groups.filter_fields',
            [
                ['name'],
                ['description', 'class' => TextareaField::class],
            ]
        );
    }

    protected function beforeSetTemplateCallback(Group $group)
    {
        $group_user_table = GroupUser::tableName();
        $options = [
            'select'     => '`from`.*',
            'joins'      => "LEFT JOIN {$group_user_table} AS gu ON gu.user_id = `from`.id",
            'conditions' => [
                '(gu.group_id <> :group OR gu.group_id IS NULL) AND `from`.id <> 1',
                'group' => $group->id,
            ],
        ];
        $available_users = Account::all($options);

        $group_users = $group->getUsers();

        // elenchi in json per la gestione delle tabelle via javascript
        $group_users_json = [];
        foreach ($group_users as $user) {
            $group_users_json[$user->user_id] = [
                'user_id' => $user->user_id,
                'name'    => $user->user->name() ?: '&nbsp;',
                'company' => $user->user->company ?: '&nbsp;',
                'email'   => $user->user->email ?: '&nbsp;',
            ];
        }

        $available_users_json = [];
        /** @var Account $user */
        foreach ($available_users as $user) {
            if (array_key_exists($user->id, $group_users_json)) {
                continue;
            }

            $available_users_json[$user->id] = [
                'user_id' => $user->id,
                'name'    => $user->name() ?: '&nbsp;',
                'company' => $user->company ?: '&nbsp;',
                'email'   => $user->email ?: '&nbsp;',
            ];
        }

        $users_json = json_encode(['inGroup' => $group_users_json, 'noGroup' => $available_users_json]);

        $url = $this->url("/groups/move/{$group->id}/");

        $js = <<<JS
(function($) {

	var users = $users_json,
		ingroup = $('#inGroup'),
		nogroup = $('#noGroup'),
		tpl = '<tr><td class="ckb"><input type="checkbox" name="users[{user_id}]" /></td><td class="main">{name}<span>{company}</span></td><td class="large">{email}</td></tr>';

	users.inGroup = users.inGroup || {};
	users.inGroup = users.inGroup.constructor == Array ? {} : users.inGroup;

	users.noGroup = users.noGroup || {};
	users.noGroup = users.noGroup.constructor == Array ? {} : users.noGroup;

	$.fn.outputTable = function(data, tpl) {
		var table = $(this),
			rows = '';

		$.each(data, function(index, row) {
			var code = tpl;

			$.each(row, function(field, value) {
				code = code.replace(new RegExp('{' + field + '}', 'g'), value);
			});

			rows += code;
		});

		table.find('.inner-table tbody').html(rows);
	};

	ingroup.outputTable(users.inGroup, tpl);
	nogroup.outputTable(users.noGroup, tpl);


	$("#group-remove").click(function(){
		var c = $(".inner-table input:checked", ingroup), movedUsers = [];

		if ( c.length )
		{
			c.each(function(){
				var name = $(this).attr('name'), id;

				// ricavo l'id
				id = name.substr(6, name.length - 7);

				movedUsers.push(id);

				// sposto la riga nella colonna di destra
				users.noGroup[id] = users.inGroup[id];
				delete users.inGroup[id];
			});

			updateTables({
				'direction'	: 'remove',
				'users'		: movedUsers
			});
		}
	});

	$("#group-insert").click(function(){
		var c = $(".inner-table input:checked", nogroup), movedUsers = [];

		if ( c.length )
		{
			c.each(function(){
				var name = $(this).attr('name');

				// ricavo l'id
				id = name.substr(6, name.length - 7);

				movedUsers.push(id);

				// sposto la riga nella colonna di sinistra
				users.inGroup[id] = users.noGroup[id];
				delete users.noGroup[id];
			});

			updateTables({
				'direction'	: 'insert',
				'users'		: movedUsers
			});
		}
	});

	function updateTables(movedData) {
		$.post('$url', movedData, function(d) {
			if (d.message === 'OK') {
				ingroup.outputTable(users.inGroup, tpl);
				nogroup.outputTable(users.noGroup, tpl);
			}
		});
	}

})(jQuery);
JS;

        $this->getHelper()->getBodyView()
            ->assignVars([
                'users'           => $group_users,
                'available_users' => $available_users,
            ]);

        $this->getHelper()
            ->addCss(pongho_url(
                '/Application/Admin/Resources/views/css/style-legacy.css?v='
                . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/css/style-legacy.css')
            ))
            ->addJavaScriptInline($js);
    }

    /**
     * @return \Pongho\Http\JsonResponse
     * @throws \ActiveRecord\Exceptions\ActiveRecordException
     */
    public function moveAction()
    {
        if ($this->getRequest()->getMethod() === 'POST') {
            if (!isset($_POST['direction']) || !isset($_POST['users']) || !is_array($_POST['users'])) {
                //Precondition Failed
                throw new HttpException(412);
            }

            $group_id = $this->getParameter('id');

            if (!$group_id) {
                throw new HttpException(404);
            }

            switch ($_POST['direction']) {
                case 'insert':
                    foreach ($_POST['users'] as $user) {
                        GroupUser::create(['group_id' => $group_id, 'user_id' => $user]);
                    }

                    break;

                case 'remove':
                    foreach ($_POST['users'] as $user) {
                        $user = GroupUser::find($group_id, $user);
                        $user->delete();
                    }

                    break;
            }

            return $this->getHelper()->displayJsonMessage('OK');
        }

        //Method Not Allowed
        throw new HttpException(405);
    }

    /**
     * Azione `moveup`.
     *
     * @return \Pongho\Http\Response
     */
    public function moveupAction()
    {
        /** @var \Application\Core\Model\Manager\GroupManager $manager */
        $manager = $this->getContainer()->get('group_manager');

        return $this->move($manager->findById($this->getParameter('id', null, true)), -15);
    }

    /**
     * Azione `movedown`.
     *
     * @return \Pongho\Http\Response
     */
    public function movedownAction()
    {
        /** @var \Application\Core\Model\Manager\GroupManager $manager */
        $manager = $this->getContainer()->get('group_manager');

        return $this->move($manager->findById($this->getParameter('id', null, true)), 15);
    }

    /**
     * Permette lo spostamento del gruppo.
     *
     * @param int   $direction
     * @return \Pongho\Http\Response
     */
    protected function move(Group $group, $direction)
    {
        $group->position += $direction;
        $group->save();

        /** @var Group $group */
        $position = 10;
        foreach (Group::all(['order' => 'position ASC']) as $group) {
            $group->position = $position;
            $group->save();

            $position += 10;
        }

        return $this->getHelper()->redirectResponse(
            $this->url('/' . $this->getParameter('path') . '/')
        );
    }
}
