<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Console\Command;

use Pongho\DependencyInjection\Container;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * @link https://confluence.jetbrains.com/display/PhpStorm/PhpStorm+Advanced+Metadata
 */
final class PhpStormMetaBuilderCommand extends Command
{
    /**
     * @var Container
     */
    private $container;

    public function __construct(Container $container)
    {
        $this->container = $container;

        parent::__construct();
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('phpstorm:meta-builder')
            ->setDescription('Builds the .phpstorm.meta.php folder');
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        if (!defined('ROOT_PATH')) {
            throw new \RuntimeException('The ROOT_PATH constant is not defined');
        }

        if (!file_exists(ROOT_PATH . '/.phpstorm.meta.php')) {
            if (!is_writable(ROOT_PATH)) {
                throw new \RuntimeException(
                    'The ROOT_PATH is not writable. Please, create the folder ".phpstorm.meta.php"'
                );
            }

            @mkdir(ROOT_PATH . '/.phpstorm.meta.php', 0o755);
        }

        if (!is_writable(ROOT_PATH . '/.phpstorm.meta.php')) {
            throw new \RuntimeException(
                'The ".phpstorm.meta.php" folder is not writable. Please, change permissions to 0644.'
            );
        }

        $map = [];
        foreach ($this->container->getServicesNames() as $serviceName) {
            $isDeprecated = false;
            try {
                $errorHandler = set_error_handler(function () use (&$isDeprecated): void {
                    $isDeprecated = true;
                }, E_USER_DEPRECATED);

                $service = $this->container->get($serviceName);

                set_error_handler($errorHandler);
            } catch (\Exception) {
                continue;
            }

            if ($isDeprecated) {
                continue;
            }

            if (is_object($service)) {
                $reflection = new \ReflectionObject($service);

                $excludedInterfaces = [
                    \ActiveRecord\ModelInterface::class,
                    \Application\Core\Entity\SettingsInterface::class,
                    \ArrayAccess::class,
                    \Countable::class,
                    \IteratorAggregate::class,
                    \Traversable::class,
                ];

                $interfaces = [];
                foreach ($reflection->getInterfaces() as $interface) {
                    if (in_array($interface->getName(), $excludedInterfaces)) {
                        continue;
                    }

                    $interfaces[] = $interface;
                }

                if ($interfaces) {
                    $interface = reset($interfaces);
                    $map[$serviceName] = "'{$serviceName}' => \\{$interface->getName()}::class,";
                } else {
                    $map[$serviceName] = "'{$serviceName}' => \\{$reflection->getName()}::class,";
                }

                continue;
            }

            $map[$serviceName] = "'{$serviceName}' => '" . gettype($service) . "',";
        }

        ksort($map);
        $map = implode("\n\t\t\t", $map);

        $code = <<<PHP
<?php

/**
 * This file is auto generated.
 *
 * Run this command for update it:
 *
 * $ php pongho/src/console phpstorm:meta-builder
 */

namespace PHPSTORM_META {
    override(
        \Pongho\DependencyInjection\Container::get(0),
        map([
            {$map}
        ])
    );

    override(
        \Pongho\DependencyInjection\Container::extend(0),
        map([
            {$map}
        ])
    );
}

PHP;

        file_put_contents(ROOT_PATH . '/.phpstorm.meta.php/di.meta.php', $code);

        return Command::SUCCESS;
    }
}
