<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Breadcrumb;

class Breadcrumb implements BreadcrumbInterface
{
    /**
     * @var CrumbInterface[]
     */
    protected $crumbs = [];

    /**
     * @var array
     */
    protected $options = [
        'after'           => '',
        'before'          => '',
        'separator'       => '›',
        'schema'          => false,
        'schema_position' => 1,
        'li_class'        => '',
        'li_active_class' => 'active',
        'link_class'      => '',
        'name_class'      => '',
        'separator_class' => 'separator',
    ];

    /**
     * {@inheritdoc}
     */
    public function append($crumb)
    {
        if ($crumb instanceof CrumbInterface) {
            $this->crumbs[$crumb->getId()] = $crumb;

            return $this;
        }

        if ($crumb instanceof BreadcrumbInterface) {
            foreach ($crumb->all() as $child) {
                $this->append($child);
            }

            return $this;
        }

        throw new \LogicException(
            'Argument 1 passed to Breadcrumb::append() must be an instance of CrumbInterface or BreadcrumbInterface'
        );
    }

    /**
     * {@inheritdoc}
     */
    public function remove($crumb_id)
    {
        unset($this->crumbs[$crumb_id]);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function all()
    {
        return $this->crumbs;
    }

    /**
     * {@inheritdoc}
     */
    public function setOptions(array $options = [])
    {
        $this->options = array_merge($this->options, $options);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function __toString(): string
    {
        return $this->render();
    }

    /**
     * @return string
     */
    public function render()
    {
        $code = '';
        $i = 1;
        $length = count($this->crumbs);
        $position = $this->options['schema_position'];

        foreach ($this->crumbs as $crumb) {
            $li_attributes = '';
            $last = $i === $length;

            if ($this->options['schema']) {
                $li_attributes .= ' itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"';
            }

            $li_class = [];

            if ($this->options['li_class']) {
                $li_class[] = $this->options['li_class'];
            }

            if ($last && $this->options['li_active_class']) {
                $li_class[] = $this->options['li_active_class'];
            }

            if ($li_class) {
                $li_attributes .= ' class="' . implode(' ', $li_class) . '"';
            }

            $item = $this->renderCrumb($crumb, $last);

            if (!$last && $this->options['separator']) {
                $separator_attributes = ' aria-hidden="true"';

                if ($this->options['separator_class']) {
                    $separator_attributes .= ' class="' . $this->options['separator_class'] . '"';
                }

                $item .= '<span' . $separator_attributes . '>' . $this->options['separator'] . '</span>';
            }

            if ($this->options['schema']) {
                $item .= '<meta itemprop="position" content="' . $position . '" />';
            }

            $code .= '<li' . $li_attributes . '>' . $item . '</li>';

            $i++;
            $position++;
        }

        return $code;
    }

    /**
     * @param CrumbInterface $crumb
     * @param bool           $last
     * @return string
     */
    protected function renderCrumb($crumb, $last = false)
    {
        $name = $this->options['before'] . $crumb->getTitle() . $this->options['after'];

        $name_attributes = '';

        if ($this->options['schema']) {
            $name_attributes .= ' itemprop="name"';
        }

        if ($this->options['name_class']) {
            $name_attributes .= ' class="' . $this->options['name_class'] . '"';
        }

        if ($name_attributes) {
            $name = '<span' . $name_attributes . '>' . $name . '</span>';
        }

        if ($last) {
            return $name;
        }

        $link_attributes = ' href="' . $crumb->getPermalink() . '"';

        if ($this->options['schema']) {
            $link_attributes .= ' itemprop="item"';
        }

        if ($this->options['link_class']) {
            $link_attributes .= ' class="' . $this->options['link_class'] . '"';
        }

        return '<a' . $link_attributes . '>' . $name . '</a>';
    }
}
