<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core;

use Guzzle\Http\Client;
use Guzzle\Http\Exception\ClientErrorResponseException;
use Pongho\Http\JsonResponse;

class BaseApiController extends Controller
{
    /**
     * {@inheritdoc}
     */
    protected function handleInvalidResponse($response): JsonResponse
    {
        return new JsonResponse($response);
    }

    /**
     * Controlla se l'utente è loggato
     *
     * @return null|\Pongho\Http\JsonResponse
     */
    protected function checkLogIn()
    {
        // Utente loggato?
        if (!$this->getUser()->isLogged()) {
            $json = [
                'error'   => true,
                'message' => $this->getLocalization()->get('api_login_required'),
            ];

            return new JsonResponse($json, [], 401);
        }

        // Permesso consentito?
        if (!$this->getUser()->hasPermit('admin')) {
            $json = [
                'error'   => true,
                'message' => $this->getLocalization()->get('api_admin_permit_required'),
            ];

            return new JsonResponse($json, [], 401);
        }

        return null;
    }

    /**
     * Consente l'accesso alle API verificando la validità del token di accesso
     *
     * @param string $app_secret
     * @return null|\Pongho\Http\JsonResponse
     */
    protected function checkAuth($app_secret)
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        if (is_labs()) {
            $auth_host = 'http://labs.metaline.it';
            $auth_path = '/pongho/auth/check_token';
        } else {
            $auth_host = 'https://serverline.metaline.it';
            $auth_path = '/auth/check_token';
        }

        $client = new Client($auth_host);

        $options = [
            'query' => [
                'ip'  => $this->getRequest()->getClientIp(),
                'app' => $app_secret,
            ],
            'auth'  => [$this->getRequest()->getUser(), $this->getRequest()->getPassword(), 'Basic'],
        ];

        $request = $client->get($auth_path, [], $options);

        $logger->info(
            'Check API authorization.',
            [
                'options' => $options,
                'request' => $request->__toString(),
            ]
        );

        try {
            $request->send();
            $response = $request->getResponse();
        } catch (ClientErrorResponseException $clientErrorResponseException) {
            $response = $clientErrorResponseException->getResponse();

            $logger->error(
                'Could not authorize the API request!',
                [
                    'request'   => $request->__toString(),
                    'response'  => $response->__toString(),
                    'exception' => $clientErrorResponseException,
                ]
            );
        }

        if ($response->isError()) {
            $logger->info(
                'The authorization response has an error status code.',
                [
                    'request'  => $request->__toString(),
                    'response' => $response->__toString(),
                ]
            );

            return new JsonResponse($response->json(), [], $response->getStatusCode());
        }

        return null;
    }
}
