<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Application;

use Application\Core\Application\Exception\CommandException;
use Application\Core\Model\Application;
use Application\Core\Model\Manager\ApplicationManagerInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

class Facade implements FacadeInterface
{
    /**
     * @var FinderInterface
     */
    protected $finder;

    /**
     * @var LoaderInterface
     */
    protected $loader;

    /**
     * @var ApplicationManagerInterface
     */
    protected $manager;

    /**
     * @var TranslatorInterface
     */
    protected $translator;

    /**
     * @var array
     */
    protected static $special_apps = ['Admin', 'Core'];

    public function __construct(
        ApplicationManagerInterface $manager,
        LoaderInterface $loader,
        FinderInterface $finder,
        TranslatorInterface $translator
    ) {
        $this->manager = $manager;
        $this->loader = $loader;
        $this->finder = $finder;
        $this->translator = $translator;
    }

    /**
     * @return \Application\Core\Model\Application[]
     */
    public function all()
    {
        $apps = $this->manager->allInstalled();

        foreach ($this->finder->find() as $app_name) {
            if (!$this->manager->isInstalled($app_name)) {
                $apps[$app_name] = new Application([
                    'app_name' => $app_name,
                ]);
            }
        }

        return $apps;
    }

    /**
     * @param string $app_name
     * @return bool
     */
    public function isInstalled($app_name)
    {
        return $this->manager->isInstalled($app_name);
    }

    /**
     * @param string                                     $app_name
     * @param int|\Application\Core\Entity\SiteInterface $site
     * @return bool
     */
    public function isEnabled($app_name, $site)
    {
        return $this->manager->isEnabled($app_name, $site);
    }

    /**
     * @param string                                     $app_name
     * @param int|\Application\Core\Entity\SiteInterface $site
     * @throws CommandException
     */
    public function install($app_name, $site)
    {
        $validator = $this->getCouldBeInstalledValidator($app_name);

        if (!$validator->isValid()) {
            throw new CommandException(
                $this->translator->trans(
                    'The application "%app%" could not be installed because: %message%',
                    [
                        '%app%'     => $app_name,
                        '%message%' => $validator->motivation(),
                    ]
                )
            );
        }

        $this->manager->createApplication($app_name);
        $this->loader->getKernel($app_name)->install();

        $this->enable($app_name, $site);
    }

    /**
     * @param string $app_name
     * @throws CommandException
     */
    public function uninstall($app_name)
    {
        $validator = $this->getCouldBeUninstalledValidator($app_name);

        if (!$validator->isValid()) {
            throw new CommandException(
                $this->translator->trans(
                    'The application "%app%" could not be uninstalled because: %message%',
                    [
                        '%app%'     => $app_name,
                        '%message%' => $validator->motivation(),
                    ]
                )
            );
        }

        $this->loader->getKernel($app_name)->uninstall();
        $this->manager->deleteApplication($app_name);
    }

    /**
     * @param string                                     $app_name
     * @param int|\Application\Core\Entity\SiteInterface $site
     * @throws Exception\CommandException
     */
    public function enable($app_name, $site)
    {
        $validator = $this->getCouldBeEnabledValidator($app_name);

        if (!$validator->isValid()) {
            throw new CommandException(
                $this->translator->trans(
                    'The application "%app%" could not be enabled because: %message%',
                    [
                        '%app%'     => $app_name,
                        '%message%' => $validator->motivation(),
                    ]
                )
            );
        }

        $this->manager->createApplicationSite($app_name, $site);
        $this->loader->getKernel($app_name)->enable();
    }

    /**
     * @param string                                     $app_name
     * @param int|\Application\Core\Entity\SiteInterface $site
     * @throws CommandException
     */
    public function disable($app_name, $site)
    {
        $validator = $this->getCouldBeDisabledValidator($app_name);

        if (!$validator->isValid()) {
            throw new CommandException(
                $this->translator->trans(
                    'The application "%app%" could not be disabled because: %message%',
                    [
                        '%app%'     => $app_name,
                        '%message%' => $validator->motivation(),
                    ]
                )
            );
        }

        $this->loader->getKernel($app_name)->disable();
        $this->manager->deleteApplicationSite($app_name, $site);
    }

    /**
     * Restituisce un oggetto che indica se l'applicazione può essere disabilitata e l'eventuale motivazione
     *
     * @param $app_name
     * @return OperationValidator
     */
    public function getCouldBeDisabledValidator($app_name)
    {
        if (in_array($app_name, static::$special_apps)) {
            return new OperationValidator(false, $this->translator->trans('The application is special'));
        }

        $deps = $this->requiredBy($app_name);

        if (!empty($deps)) {
            return new OperationValidator(
                false,
                $this->translator->trans(
                    'Other applications depend from it [ %deps% ]',
                    ['%deps%' => implode(', ', $deps)]
                )
            );
        }

        if (!$this->loader->kernelExists($app_name)) {
            return new OperationValidator(false, $this->translator->trans('The application kernel does not exist'));
        }

        return new OperationValidator(true);
    }

    /**
     * Restituisce un oggetto che indica se l'applicazione può essere abilitata e l'eventuale motivazione
     *
     * @param $app_name
     * @return OperationValidator
     */
    public function getCouldBeEnabledValidator($app_name)
    {
        if (in_array($app_name, static::$special_apps)) {
            return new OperationValidator(false, $this->translator->trans('The application is special'));
        }

        $deps = $this->dependsFrom($app_name);

        if (!empty($deps)) {
            return new OperationValidator(
                false,
                $this->translator->trans(
                    'Missing dependencies [ %deps% ]',
                    ['%deps%' => implode(', ', $deps)]
                )
            );
        }

        if (!$this->loader->kernelExists($app_name)) {
            return new OperationValidator(false, $this->translator->trans('The application kernel does not exist'));
        }

        return new OperationValidator(true);
    }

    /**
     * Restituisce un oggetto che indica se l'applicazione può essere disinstallata e l'eventuale motivazione
     *
     * @param $app_name
     * @return OperationValidator
     */
    public function getCouldBeUninstalledValidator($app_name)
    {
        if (in_array($app_name, static::$special_apps)) {
            return new OperationValidator(false, $this->translator->trans('The application is special'));
        }

        $deps = $this->requiredBy($app_name);

        if (!empty($deps)) {
            return new OperationValidator(
                false,
                $this->translator->trans(
                    'Other applications depend from it [ %deps% ]',
                    ['%deps%' => implode(', ', $deps)]
                )
            );
        }

        // Non posso disinstallare un'applicazione usata da un altro sito
        if ($app = $this->manager->findByAppName($app_name)) {
            if (!$this->loader->kernelExists($app_name)) {
                return new OperationValidator(
                    false,
                    $this->translator->trans('The application kernel does not exist')
                );
            }

            if ($app->total_installations !== 0) {
                return new OperationValidator(
                    false,
                    $this->translator->trans('The application is enabled in another site')
                );
            }

            return new OperationValidator(true);
        }

        return new OperationValidator(false, $this->translator->trans('The application does not exist'));
    }

    public function getCouldBeInstalledValidator($app_name)
    {
        if (in_array($app_name, static::$special_apps)) {
            return new OperationValidator(false, $this->translator->trans('The application is special'));
        }

        if (!$this->manager->findByAppName($app_name)) {
            if ($this->loader->kernelExists($app_name)) {
                $deps = $this->dependsFrom($app_name);

                if (!empty($deps)) {
                    return new OperationValidator(
                        false,
                        $this->translator->trans(
                            'Missing dependencies [ %deps% ]',
                            ['%deps%' => implode(', ', $deps)]
                        )
                    );
                }

                return new OperationValidator(true);
            }

            return new OperationValidator(false, $this->translator->trans('The application kernel does not exist'));
        }

        return new OperationValidator(false, $this->translator->trans('The application does not exist'));
    }

    /**
     * @param $app_name
     * @return array
     */
    protected function dependsFrom($app_name)
    {
        $kernel_class = $this->loader->getKernelClass($app_name);

        $deps = [];
        foreach ($kernel_class::depends() as $app) {
            if (!$this->manager->isInstalled($app)) {
                $deps[] = $app;
            }
        }

        return $deps;
    }

    /**
     * @param $app_name
     * @return array
     */
    protected function requiredBy($app_name)
    {
        $dependencies = [];
        foreach ($this->manager->allInstalled() as $app) {
            $kernel_class = $this->loader->getKernelClass($app->getName());

            if (in_array($app_name, $kernel_class::depends())) {
                $dependencies[] = $app->getName();
            }
        }

        return $dependencies;
    }
}
