<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core\Utilities;

/**
 * Gestisce gli shortcode.
 */
class Shortcode
{
	/**
	 * Lista dei tag.
	 *
	 * @access protected
	 * @var array
	 */
	protected $tags = array();

	/**
	 * Aggiunge un tag alla lista.
	 *
	 * Il parametro $callback può essere qualsiasi cosa soddisfi la funzione ({@link http://php.net/manual/en/function.is-callable.php is_callable}).
	 *
	 * @access public
	 * @param string $tag_name Nome del tag.
	 * @param mixed $callback La callback da usare per il parsing dello shortcode.
	 * @return Shortcode La classe stessa.
	 */
	public function add($tag_name, $callback)
	{
		if ( is_callable($callback) )
		{
			$this->tags[$tag_name] = $callback;
		}

		return $this;
	}

	/**
	 * Rimuove un tag.
	 *
	 * @access public
	 * @param string $tag_name Nome del tag da rimuovere.
	 * @return Shortcode La classe stessa.
	 */
	public function remove($tag_name)
	{
		unset($this->tags[$tag_name]);

		return $this;
	}

	/**
	 * Rimuove tutti i tag.
	 *
	 * @access public
	 * @return Shortcode La classe stessa.
	 */
	public function removeAll()
	{
		$this->tags = array();

		return $this;
	}

	/**
	 * Elabora un testo con gli shortcode.
	 *
	 * @access public
	 * @param string $text Testo da elaborare.
	 * @param array $attributes Attributi aggiuntvi da passare allo shortcode.
	 * @return string Il testo elaborato.
	 */
	public function parse($text, array $attributes = array())
	{
		if ( empty($this->tags) )
		{
			return $text;
		}

		$that = $this;

		return preg_replace_callback('/' . $this->getRegex() . '/s', function($match) use ($attributes, $that) {
			return $that->parseTag($match, $attributes);
		}, $text);
	}

	/**
	 * Rimuove gli shortcode da un testo.
	 *
	 * @access public
	 * @param string $text Testo da elaborare.
	 * @return string Il testo senza shortcode.
	 */
	public function strip($text)
	{
		if ( empty($this->tags) )
		{
			return $text;
		}

		return preg_replace('/' . $this->getRegex() . '/s', '', $text);
	}

	/**
	 * Metodo interno di callback usato dal metodo parse().
	 *
	 * La visibilità è impostata a public in quanto questo metodo è richiamato attraverso una funzione
	 * anonima, ma non è il caso di usarlo fuori da questa classe.
	 *
	 * @access public
	 * @param array $match Risultato dell’espressione regolare sul testo da elaborare.
	 * @param array $attributes Attributi aggiuntivi.
	 * @return string
	 *
	 * @ignore
	 */
	public function parseTag(array $match, array $attributes = array())
	{
		$tag_name = $match[3];
		$attributes = array_merge($attributes, $this->parseAttributes($match[4]));
		$content = isset($match[7]) ? $match[7] : '';

		return call_user_func($this->tags[$tag_name], $attributes, $content, $tag_name);
	}

	/**
	 * Restituisce l'espressione regolare per il parsing del testo.
	 *
	 * Deve soddisfare questi codici:
	 * <code>
	 *   [tag /]
	 *   [tag attr="value" /]
	 *   [tag] ... [/tag]
	 *   [tag attr="value"] ... [/tag]
	 * </code>
	 *
	 * @access protected
	 * @return string L’espressione regolare per la ricerca degli shortcode.
	 */
	protected function getRegex()
	{
		$tagnames = array_keys($this->tags);
		$tagregexp = implode('|', array_map('preg_quote', $tagnames));

		return '(<(p)>)?\[(' . $tagregexp . ')([^\\]\\/]*(?:\\/(?!\\])[^\\]\\/]*)*?)(?:( \/\])|(\](.*?)\[\/\3\]))(<\/\2>)?';
	}

	/**
	 * Converte una stringa di attributi in array.
	 *
	 * @access protected
	 * @param string $text Stringa di attributi da elaborare.
	 * @return array L’elenco degli attributi.
	 */
	protected function parseAttributes($text)
	{
		$text = trim($text);

		if ( empty($text) )
		{
			return array();
		}

		preg_match_all('/(.*?)=[\"\'](.*?)[\"\']/', $text, $matches, PREG_SET_ORDER);

		$attributes = array();
		foreach ( $matches as $match )
		{
			list($code, $attr_name, $attr_value) = $match;
			$attributes[trim($attr_name)] = trim($attr_value);
		}

		return $attributes;
	}
}
