<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core\Utilities;

use Application\Core\Model\Group;
use Application\Core\Model\Province;
use Application\Core\Model\Role;
use Pongho\Core\Kernel;
use Pongho\Template\Html as HtmlBase;

/**
 * Aggiunge alcune funzionalità alla classe `Pongho\Template\Html`.
 *
 * @see \Pongho\Template\Html
 */
class Html extends HtmlBase
{
	/**
	 * Istanza della classe per il singleton.
	 *
	 * @access private
	 * @var Html
	 */
	private static $instance;

	/**
	 * Restituisce l’istanza della classe (Singleton).
	 *
	 * @access public
	 * @return Html
	 * @static
	 */
	public static function instance()
	{
		if ( is_null(self::$instance) )
		{
			self::$instance = new static;
		}

		return self::$instance;
	}

	/**
	 * Genera una select con la lista delle province.
	 *
	 * @access public
	 * @param integer $province Indica la provincia da selezionare. Per il confronto viene usato il campo indicato in `$field`.
	 * @param string $field Il campo da usare come valore della select. Può essere `id`, `plate` o `name`.
	 * @return string Il codice HTML della select.
	 */
	public function selectProvince($province_id = null, $field = 'id')
	{
		$provinces = Province::all(array('order' => 'name'));

		$options = array();
		foreach ( $provinces as $province )
		{
			$options[$province->{$field}] = $province->name;
		}

		return $this->select($options, $province_id);
	}

	/**
	 * Genera una select con la lista dei gruppi.
	 *
	 * @access public
	 * @param integer $group_id ID del gruppo da selezionare.
	 * @return string Il codice HTML della select.
	 */
	public function selectGroup($group_id = null)
	{
		$groups = Group::all(Group::get_default_options());

		$options = array();
		foreach ( $groups as $group )
		{
			$options[$group->id] = $group->name;
		}

		return $this->select($options, $group_id);
	}

	/**
	 * Genera una select con la lista dei ruoli.
	 *
	 * @access public
	 * @param integer $role_id ID del ruolo da selezionare.
	 * @param array $excludes_roles Permette di escludere alcuni ruoli dalla select.
	 * @return string Il codice HTML della select.
	 */
	public function selectRole($role_id = null, array $excludes_roles = array())
	{
		$roles = Role::all(Role::get_default_options());

		$options = array();
		foreach ( $roles as $role )
		{
			if ( !in_array($role->id, $excludes_roles) )
			{
				$options[$role->id] = $role->name;
			}
		}

		return $this->select($options, $role_id);
	}

	/**
	 * Genera una select con la lista delle lingue.
	 *
	 * @access public
	 * @param integer $language_id ID della lingua da selezionare.
	 * @return string Il codice HTML della select.
	 */
	public function selectLanguage($language_id = null)
	{
		$site = Kernel::instance()->getContainer()->get('site');

		$options = array();
		foreach ( $site->getLanguages() as $language )
		{
			$options[$language->language_id] = $language->name;
		}

		return $this->select($options, $language_id);
	}
}
