<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Utilities;

use Application\Core\Model\Account;
use Application\Core\Notification\NotificationCollection;
use Application\Core\Notification\NotificationInterface;
use Application\Core\Notification\NotificationSender;

/**
 * AdminUserNotificationHelper
 */
class AdminUserNotificationHelper
{
    /**
     * @var NotificationInterface[]
     */
    protected $notifications = array();

    /**
     * @var NotificationSender
     */
    protected $sender;

    /**
     * @var NotificationCollection
     */
    protected $collection;

    /**
     * @var UserActivationLinkBuilder
     */
    protected $activation_link_builder;

    /**
     * @param NotificationSender        $sender
     * @param NotificationCollection    $collection
     * @param UserActivationLinkBuilder $builder
     */
    public function __construct(NotificationSender $sender, NotificationCollection $collection, UserActivationLinkBuilder $builder)
    {
        $this->sender = $sender;
        $this->collection = $collection;
        $this->activation_link_builder = $builder;
    }

    /**
     * @param Account $account
     * @return bool
     */
    public function addUserEditNotification(Account $account)
    {
        if ($account->isActive() === true) {
            /** @var \Application\Core\Notification\UserStatusActiveNotification $notification */
            $notification = $this->collection->get('core.user_edit');
            $notification->setAccount($account);

            $this->addNotification($notification);

            return true;
        }

        return false;
    }

    /**
     * @param Account $account
     * @return bool
     */
    public function addUserStatusChangeNotification(Account $account)
    {
        if ($account->isActive()) {
            $this->addUserStatusActivateNotification($account);

            return true;
        } elseif ($account->isActive() === false) {
            $this->addUserStatusDeactivateNotification($account);

            return true;
        }

        return false;
    }

    /**
     * @param Account $account
     * @param string  $password
     * @return bool
     */
    public function addUserCredentialsNotification(Account $account, $password)
    {
        if ($account->isActive() === true) {
            /** @var \Application\Core\Notification\SendCredentialsNotification $notification */
            $notification = $this->collection->get('core.user_send_credentials');
            $notification->setAccount($account);
            $notification->setPassword($password);

            $this->addNotification($notification);

            return true;
        }

        return false;
    }

    /**
     * @param Account $account
     * @return bool
     */
    public function addUserActivationLinkNotification(Account $account)
    {
        if ($account->isActive() === null) {
            /** @var \Application\Core\Notification\SendActivationLinkNotification $notification */
            $notification = $this->collection->get('core.user_send_activation_link');
            $notification->setAccount($account);
            $notification->setAccountActivationLink($this->activation_link_builder->build($account));

            $this->addNotification($notification);

            return true;
        }

        return false;
    }

    /**
     * @param Account $account
     * @param string  $password
     * @return bool
     */
    public function addUserCreationNotification(Account $account, $password)
    {
        if ($account->isActive()) {
            $this->addUserActiveCreationNotification($account, $password);

            return true;
        } elseif ($account->isActive() === null) {
            $this->addUserNotActiveCreationNotification($account, $password);

            return true;
        }

        return false;
    }

    /**
     * @param Account $account
     */
    protected function addUserStatusActivateNotification(Account $account)
    {
        /** @var \Application\Core\Notification\UserStatusActiveNotification $notification */
        $notification = $this->collection->get('core.user_status_active');
        $notification->setAccount($account);

        $this->addNotification($notification);
    }

    /**
     * @param Account $account
     */
    protected function addUserStatusDeactivateNotification(Account $account)
    {
        /** @var \Application\Core\Notification\UserStatusNotActiveNotification $notification */
        $notification = $this->collection->get('core.user_status_not_active');
        $notification->setAccount($account);

        $this->addNotification($notification);
    }

    /**
     * @param Account $account
     * @param string  $password
     */
    protected function addUserActiveCreationNotification(Account $account, $password)
    {
        /** @var \Application\Core\Notification\UserCreationNotification $notification */
        $notification = $this->collection->get('core.user_create_active');
        $notification->setAccount($account);
        $notification->setPassword($password);

        $this->addNotification($notification);
    }

    /**
     * @param Account $account
     * @param string  $password
     */
    protected function addUserNotActiveCreationNotification(Account $account, $password)
    {
        /** @var \Application\Core\Notification\UserCreationActivationNotification $notification */
        $notification = $this->collection->get('core.user_create_pending');
        $notification->setAccount($account);
        $notification->setPassword($password);
        $notification->setAccountActivationLink($this->activation_link_builder->build($account));

        $this->addNotification($notification);
    }

    /**
     * @param NotificationInterface $notification
     */
    protected function addNotification(NotificationInterface $notification)
    {
        $this->notifications[] = $notification;
    }

    /**
     * @return $this
     * @throws \Application\Core\Notification\Exception\SendException
     */
    public function send()
    {
        foreach ($this->notifications as $notification) {
            $this->sender->send($notification);
        }

        return $this;
    }
}
