<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Tests\Utilities\Link;


use Application\Core\Entity\LinkTypeDetails;
use Application\Core\Tests\Utilities\Link\Utilities\Mocks;
use Application\Core\Utilities\Html;
use Application\Core\LinkType\Handler\ModelHandler;
use Application\Core\LinkType\LinkSelectOption;
use Symfony\Component\DomCrawler\Crawler;

/**
 * Class ModelHandlerTest
 */
class ModelHandlerTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var Html
     */
    protected $html;

    /**
     * @var Mocks
     */
    protected $mocks;

    public function setUp()
    {
        $this->html = Html::instance();
        $this->mocks = new Mocks($this);
    }

    public function testInitialization()
    {
        $expected_name = 'Name';
        $expected_label = 'Label';
        $options = array();

        $handler = $this->getHandler($expected_name, $expected_label, $options);

        $this->assertEquals($expected_name, $handler->getName());
        $this->assertEquals($expected_label, $handler->getLabel());
    }

    public function testGetDetailsFromPost()
    {
        $handler = $this->getHandler('model', '', array());

        $expected_language_id = 123;
        $expected_model_id = 321;

        $post = array(
            'language_id' => $expected_language_id,
            'model'       => $expected_model_id,
        );

        $details = $handler->getDetails($post);

        $this->assertInstanceOf('Application\\Core\\Entity\\LinkTypeDetails', $details);
        $this->assertEquals($expected_language_id, $details->getLanguageId());
        $this->assertEquals($expected_model_id, $details->getModelId());
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testGetDetailsThrowsExceptionForMissingLanguageInPost()
    {
        $handler = $this->getHandler('', '', array());

        $post = array(
            'model_id' => 123,
        );

        $handler->getDetails($post);
    }

    /**
     * @expectedException \InvalidArgumentException
     */
    public function testGetDetailsThrowsExceptionForMissingModelInPost()
    {
        $handler = $this->getHandler('', '', array());

        $post = array(
            'language_id' => 123,
        );

        $handler->getDetails($post);
    }

    public function testRenderSelectWithOptions()
    {
        /** @var LinkSelectOption[] $options */
        $options = $this->getLinkSelectOptions();
        $handler = $this->getHandler('test', 'Test models', $options);

        $render = $handler->render('form[fieldset][link_type]', null);

        $this->assertRenderContainsSelectWithDefinedOptions($render, $options);

        $crawler = new Crawler($render);

        /** @var \DOMElement $node */
        foreach ($crawler->filter('option') as $node) {
            if ($node->hasAttribute('value')) {
                if (!array_key_exists($node->getAttribute('value'), $options)) {
                    $this->fail('The option "' . $node->getAttribute('value') . '" could not be found in the defined array, check the test');
                }

                $label = $options[$node->getAttribute('value')]->getLabel();
                $this->assertEquals($label, $node->textContent);
            }
        }
    }

    public function testGetDefinedLink()
    {
        /** @var LinkSelectOption[] $options */
        $options = array(
            new LinkSelectOption(1, 'A model', '/path/to/model/a'),
            new LinkSelectOption(2, 'B model', '/path/to/model/b'),
            new LinkSelectOption(3, 'C model', '/path/to/model/c'),
            new LinkSelectOption(4, 'D model', '/path/to/model/d'),
            new LinkSelectOption(5, 'E model', '/path/to/model/e'),
        );

        $selected_id = 4;
        $expected_url = '/path/to/model/d';

        $handler_name = 'test';
        $handler = $this->getHandler($handler_name, '', $options);

        $post = array(
            $handler_name => $selected_id
        );

        $this->assertEquals($expected_url, $handler->getUrl($post));
    }

    public function testGetValue()
    {
        /** @var LinkSelectOption[] $options */
        $expected_value = 123;

        $handler_name = 'test';
        $handler = $this->getHandler($handler_name, '', array());

        $link_type = $this->mocks->getLinkTypeMock();
        $link_type->expects($this->any())
            ->method('getLinkDetails')
            ->willReturn(new LinkTypeDetails(null, $expected_value));
        $link_type->expects($this->any())->method('getIdentifier')->willReturn($handler_name);

        $this->assertInstanceOf('Application\\Core\\Entity\\LinkTypeInterface', $link_type);

        $this->assertEquals($expected_value, $handler->getValue($link_type));
    }

    /**
     * @param string $key
     * @param string $label
     * @param LinkSelectOption[] $models
     * @return ModelHandler
     */
    protected function getHandler($key, $label, array $models = array())
    {
        return new ModelHandler($key, $label, $models, $this->html);
    }

    /**
     * @return LinkSelectOption[]
     */
    protected function getLinkSelectOptions()
    {
        return array(
            1 => new LinkSelectOption(1, 'A model', '/path/to/model/a'),
            2 => new LinkSelectOption(2, 'B model', '/path/to/model/b'),
        );
    }

    /**
     * @param string             $render
     * @param LinkSelectOption[] $options
     */
    protected function assertRenderContainsSelectWithDefinedOptions($render, array $options)
    {
        $crawler = new Crawler($render);

        $this->assertEquals(1, $crawler->filter('select')->count());
        $this->assertEquals('form[fieldset][link_type]', $crawler->filter('select')->attr('name'));
        $this->assertEquals(count($options) + 1 /* empty placeholder */, $crawler->filter('option')->count());
    }
}
