<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Tests\Utilities\Link;

use Application\Core\Entity\LinkTypeDetails;
use Application\Core\Tests\Utilities\Link\Utilities\Mocks;
use Application\Core\Utilities\Html;
use Application\Core\LinkType\LinkFacade;
use Application\Core\LinkType\LinkHandlerCollection;
use Application\Core\LinkType\LinkHandlerInterface;
use Symfony\Component\DomCrawler\Crawler;

/**
 * Class LinkFacadeTest
 */
class LinkFacadeTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var Html
     */
    protected $html;

    /**
     * @var Mocks
     */
    protected $mocks;

    /**
     * {@inheritdoc}
     */
    public function setUp()
    {
        $this->html = Html::instance();
        $this->mocks = new Mocks($this);
    }

    public function testRenderEmptyFacade()
    {
        $name = 'test[fieldgroup]';
        $facade = $this->getEmptyFacade();

        $this->assertLinkTypeRenderOnlyLinkTypeSelectorWithoutOptions($facade, $name);
    }

    public function testFacadeRenderContainsThreeOptions()
    {
        $name = 'test[fieldgroup]';
        $facade = $this->getFacadeWithOneHandlerForEveryType();

        $this->assertLinkTypeRenderContainsLinkTypeSelectorAndThreeLinkTypes($facade, $name);
    }

    public function testFacadeRenderContainNolinkType()
    {
        $facade = $this->getFacadeWithOneHandlerForEveryType();
        $this->assertLinkTypeRendersOneDivWithDataAttributes($facade, 'nolink', 'form[fieldgroup][link_type]');
    }

    public function testFacadeRenderContainLinkType()
    {
        $facade = $this->getFacadeWithOneHandlerForEveryType();
        $this->assertLinkTypeRendersOneDivWithDataAttributes($facade, 'link', 'form[fieldgroup][link_type]');
    }

    public function testFacadeRenderContainModelType()
    {
        $facade = $this->getFacadeWithOneHandlerForEveryType();
        $this->assertLinkTypeRendersOneDivWithDataAttributes($facade, 'node_page', 'form[fieldgroup][link_type]');
    }

    public function testRenderJsonIsValidJson()
    {
        $this->assertJson($this->getFacadeWithOneHandlerForEveryType()->renderJson());
    }

    public function testJsonContainsNolinkRender()
    {
        $this->assertJsonContainsLinkTypeRender($this->getFacadeWithOneHandlerForEveryType(), 'nolink');
    }

    public function testJsonContainsLinkRender()
    {
        $this->assertJsonContainsLinkTypeRender($this->getFacadeWithOneHandlerForEveryType(), 'link');
    }

    public function testJsonContainsModelRender()
    {
        $this->assertJsonContainsLinkTypeRender($this->getFacadeWithOneHandlerForEveryType(), 'node_page');
    }

    public function testSaveForCreation()
    {
        $identifier = 'node_page';
        $url = '';
        $language_id = null;
        $model_id = 1;

        $link_type_mock = $this->mocks->getLinkTypeMock();
        $link_type_mock->expects($this->once())->method('change')->with(
            $identifier, $url, $this->callback(
                function (LinkTypeDetails $details) use ($language_id, $model_id) {
                    if ($details->getModelId() !== $model_id) {
                        return false;
                    }

                    if ($details->getLanguageId() !== $language_id) {
                        return false;
                    }

                    return true;
                }
            )
        );

        $manager = $this->mocks->getLinkTypeManagerMock();
        $manager->expects($this->any())->method('newModel')->willReturn($link_type_mock);
        $manager->expects($this->once())->method('save');

        $handler = $this->getHandlerMock('node_page', 'Pagine');
        $handler->expects($this->any())->method('getDetails')->willReturn(new LinkTypeDetails($language_id, $model_id));

        $collection = new LinkHandlerCollection();
        $collection->add($handler);

        $facade = new LinkFacade($collection, $manager, $this->html);

        $post = array(
            'link_handler' => $identifier,
            'link'         => $url,
            'node_page'    => $model_id,
        );

        $link_type = $facade->save($post, $language_id);

        $this->assertSame($link_type_mock, $link_type);
    }

    public function testRenderWithNullValue()
    {
        $identifier = 'link';
        $manager = $this->mocks->getLinkTypeManagerMock();
        $manager->expects($this->once())->method('findById')->with(null)->willReturn(null);

        $collection = new LinkHandlerCollection();

        $handler = $this->getHandlerMock($identifier, 'Label');
        $handler->expects($this->never())->method('getValue');
        $collection->add($handler);

        $facade = new LinkFacade($collection, $manager, $this->html);

        $facade->render('form');
    }

    public function testSaveForEdit()
    {
        $this->markTestIncomplete('impossibile da testare senza un\'istanza reale');

        $link_type_id = 123;
        $identifier = 'link';
        $expected_value = 'value';
        $expected_url = 'new url';

        $link_type = $this->mocks->getLinkTypeMock();
        $link_type->expects($this->any())->method('getIdentifier')->willReturn($identifier);
        $link_type->expects($this->once())->method('getDetails')->willReturn(new LinkTypeDetails(null, null));
        $link_type->expects($this->any())->method('getUrl')->willReturn($expected_value);
        $link_type->expects($this->once())->method('getValue')->willReturn($expected_value);
        $link_type->expects($this->once())->method('getId')->willReturn($link_type_id);

        $manager = $this->mocks->getLinkTypeManagerMock();
        $manager->expects($this->any())->method('findById')->with($link_type_id)->willReturn($link_type);
        $manager->expects($this->any())->method('save')->willReturn(true);

        $collection = new LinkHandlerCollection();

//        $handler = new LinkHandler($identifier, 'LABEL');
//
        $handler = $this->getHandlerMock($identifier, 'Label');
        $handler->expects($this->any())->method('getValue')->willReturn($expected_value);
        $handler->expects($this->any())->method('getDetails')->willReturn(new LinkTypeDetails(null, null));
        $collection->add($handler);

        $facade = new LinkFacade($collection, $manager, $this->html);

        $this->assertLinkTypeRenderWithValueFromEntity($facade, 'test', $link_type_id);

        $post = array(
            'link_handler' => $identifier,
            'link'         => $expected_url,
        );

        $facade->save($post, $link_type_id);

        // TODO: impossibile da testare senza un'istanza reale
        $this->assertLinkTypeRenderWithValueFromEntity($facade, 'test', $link_type_id);
    }

    public function testSaveForDeletion()
    {
        // TODO: impossibile da testare senza un'istanza reale
        $this->markTestIncomplete('impossibile da testare senza un\'istanza reale');
    }

    /**
     * @return LinkFacade
     */
    protected function getEmptyFacade()
    {
        $manager = $this->mocks->getLinkTypeManagerMock();
        $collection = new LinkHandlerCollection();

        return new LinkFacade($collection, $manager, $this->html);
    }

    /**
     * @return LinkFacade
     */
    protected function getFacadeWithOneHandlerForEveryType()
    {
        $manager = $this->mocks->getLinkTypeManagerMock();
        $collection = $this->getHandlerCollectionWithEveryType();

        return new LinkFacade($collection, $manager, $this->html);
    }

    /**
     * @param string $name
     * @param string $label
     * @param string $content
     * @return LinkHandlerInterface|\PHPUnit_Framework_MockObject_MockObject
     */
    protected function getHandlerMock($name, $label, $content = '')
    {
        $handler = $this->mocks->getLinkHandlerMock();

        $handler->expects($this->any())->method('getName')->willReturn($name);
        $handler->expects($this->any())->method('getLabel')->willReturn($label);
        $handler->expects($this->any())->method('renderOptions')->willReturn($content);

        return $handler;
    }

    /**
     * @param LinkFacade $facade
     * @param string     $type
     * @param string     $name
     * @param string     $language
     */
    protected function assertLinkTypeRendersOneDivWithDataAttributes(LinkFacade $facade, $type, $name, $language = '')
    {
        $render = $facade->render($name);
        $crawler = new Crawler($render);

        $div = $crawler->filter('.link-type-' . $type);

        $this->assertEquals(1, $div->count());
        $this->assertEquals('', $div->text());
        $this->assertEquals($name . '[' . $type . ']', $div->attr('data-name'));
        $this->assertEquals($language, $div->attr('data-language'));
    }

    /**
     * @param LinkFacade $facade
     * @param string     $type
     */
    protected function assertJsonContainsLinkTypeRender(LinkFacade $facade, $type)
    {
        $json = $facade->renderJson();
        $types = json_decode($json, true);

        $this->assertArrayHasKey($type, $types);
    }

    /**
     * @param LinkFacade $facade
     * @param            $name
     */
    protected function assertLinkTypeRenderContainsLinkTypeSelectorAndThreeLinkTypes(LinkFacade $facade, $name)
    {
        $render = $facade->render($name);
        $crawler = new Crawler($render);

        $wrapper = $crawler->filter('.link-type-wrapper');

        $this->assertEquals(1, $wrapper->filter('.link-type-select')->count());
        $this->assertEquals(3, $wrapper->filter('.link-type-select option')->count());
        $this->assertEquals(3, $wrapper->filter('.link-type')->count());
    }

    /**
     * @param LinkFacade $facade
     * @param            $name
     */
    protected function assertLinkTypeRenderOnlyLinkTypeSelectorWithoutOptions(LinkFacade $facade, $name)
    {
        $render = $facade->render($name);
        $crawler = new Crawler($render);

        $wrapper = $crawler->filter('.link-type-wrapper');
        $this->assertEquals(1, $wrapper->count());
        $this->assertEquals(1, $wrapper->filter('.link-type-select')->count());
        $this->assertEquals(0, $wrapper->filter('.link-type-select option')->count());
    }

    /**
     * @param LinkFacade $facade
     * @param string     $name
     * @param integer    $link_type_id
     */
    protected function assertLinkTypeRenderWithValueFromEntity(LinkFacade $facade, $name, $link_type_id)
    {
        $render = $facade->render($name, $link_type_id);
        $crawler = new Crawler($render);

        $wrapper = $crawler->filter('.link-type-wrapper');
        $this->assertEquals(1, $wrapper->count());
        $this->assertEquals(1, $wrapper->filter('.link-type-select')->count());
        $this->assertEquals(1, $wrapper->filter('.link-type-select option:selected')->count());
    }

    /**
     * @return LinkHandlerCollection
     */
    protected function getHandlerCollectionWithEveryType()
    {
        $collection = new LinkHandlerCollection();
        $collection->add($this->getHandlerMock('nolink', 'Nessun Link', 'nolink'));
        $collection->add($this->getHandlerMock('link', 'Link diretto', 'link'));
        $collection->add($this->getHandlerMock('node_page', 'Pagina', 'model'));
        return $collection;
    }
}
