<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Template;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\SiteInterface;
use Pongho\Http\Request;
use Pongho\Template\FooterHelper;
use Pongho\Template\HeadHelper;
use Pongho\Template\Helper\Css;
use Pongho\Template\Helper\JavaScript;
use Pongho\Template\Helper\Links;
use Pongho\Template\Helper\Metatags;

/**
 * HeadHelperBuilder
 */
class TemplateHelperBuilder
{
    /**
     * @var HeadHelper
     */
    protected $header;

    /**
     * @var FooterHelper
     */
    protected $footer;

    /**
     * @var Metatags
     */
    protected $metatags;

    /**
     * @var Links
     */
    protected $links;

    /**
     * @var Css
     */
    protected $css;

    /**
     * @var Javascript
     */
    protected $js_header;

    /**
     * @var Javascript
     */
    protected $js_footer;

    /**
     * @var Request
     */
    protected $request;

    /**
     * @var SiteInterface
     */
    protected $site;

    /**
     * @var LanguageInterface
     */
    protected $language;

    /**
     * @param Request           $request
     * @param SiteInterface     $site
     * @param LanguageInterface $language
     */
    public function __construct(Request $request, SiteInterface $site, LanguageInterface $language)
    {
        $this->request = $request;
        $this->site = $site;
        $this->language = $language;
    }

    /**
     * @return HeadHelper
     */
    public function buildHeader()
    {
        $this->header = new HeadHelper(
            $this->getMetatags(),
            $this->getLinks(),
            $this->getCss(),
            $this->getHeaderJs(),
            $this->getFooterJs()
        );

        $this->prepareHeader();

        return $this->header;
    }

    /**
     * @return FooterHelper
     */
    public function buildFooter()
    {
        $this->footer = new FooterHelper($this->getFooterJs());

        $this->prepareFooter();

        return $this->footer;
    }

    /**
     * @return Metatags
     */
    protected function getMetatags()
    {
        if ($this->metatags === null) {
            $this->metatags = new Metatags();
        }

        return $this->metatags;
    }

    /**
     * @return Links
     */
    protected function getLinks()
    {
        if ($this->links === null) {
            $this->links = new Links();
        }

        return $this->links;
    }

    /**
     * @return Css
     */
    protected function getCss()
    {
        if ($this->css === null) {
            $this->css = new Css();
        }

        return $this->css;
    }

    /**
     * @return JavaScript
     */
    protected function getHeaderJs()
    {
        if ($this->js_header === null) {
            $this->js_header = new JavaScript();
        }

        return $this->js_header;
    }

    /**
     * @return JavaScript
     */
    protected function getFooterJs()
    {
        if ($this->js_footer === null) {
            $this->js_footer = new JavaScript();
        }

        return $this->js_footer;
    }

    /**
     *
     */
    protected function prepareHeader()
    {
        // X-UA-Compatible
        $this->metatags->add('X-UA-Compatible', 'IE=edge', array('type' => 'http-equiv'));

        // viewport
        $this->metatags->add('viewport', 'width=device-width, initial-scale=1');

        // copyright
        $this->metatags->add('copyright', $this->site->getOption('company_copyright'));

        // social
        $this->prepareHeaderOpenGraph();
        $this->prepareHeaderFacebook();

        // icons
        $this->prepareFavicon();
        $this->prepareAppleTouchIcon();

        // humans.txt
        $this->prepareHumansTxt();

        // sitemap.xml
        $this->prepareSitemap();

        // cookieManager & firstUserInteraction
        $this->prepareHeaderJavascript();
    }

    /**
     *
     */
    protected function prepareFooter()
    {
    }

    /**
     * Open Graph
     *
     * @see http://ogp.me/
     */
    protected function prepareHeaderOpenGraph()
    {
        $this->metatags->add(
            'og:title',
            function () {
                return $this->header->getTitle();
            },
            array('type' => 'property')
        );

        $this->metatags->add(
            'og:description',
            function () {
                return $this->header->metatags->getContent('description');
            },
            array('type' => 'property')
        );

        $this->metatags->add('og:type', $this->site->getOption('fb_og_type'), array('type' => 'property'));
        $this->metatags->add('og:site_name', $this->site->getOption('site_name'), array('type' => 'property'));
        $this->metatags->add('og:url', $this->site->getPermalink() . ltrim($this->request->getPathInfo(), '/'), array('type' => 'property'));
        $this->metatags->add('og:locale', $this->language->getCulture(), array('type' => 'property'));

        // TODO: L’immagine per "og:image" dovrebbe essere caricata da interfaccia
        $this->metatags->add(
            'og:image',
            $this->site->getThemesUrl('/images/favicons/apple-touch-icon.png'),
            array('type'  => 'property')
        );
    }

    /**
     * Facebook API
     */
    protected function prepareHeaderFacebook()
    {
        if ($this->site->getOption('fb')) {
            $this->metatags->add('fb:admins', $this->site->getOption('fb_admin_id'), array('type' => 'property'));
            $this->metatags->add('fb:app_id', $this->site->getOption('fb_appid'), array('type' => 'property'));
        }
    }

    /**
     * Favicon
     *
     * @todo Unit Test
     */
    protected function prepareFavicon()
    {
        $file = '/images/favicons/favicon.ico';

        if (file_exists($this->site->getThemesPath($file))) {
            $this->links->add(
                $this->site->getThemesUrl($file),
                array(
                    'rel'  => 'shortcut icon',
                    'type' => 'image/x-icon',
                )
            );
        }
    }

    /**
     * Apple Touch Icon
     *
     * @todo Unit Test
     */
    protected function prepareAppleTouchIcon()
    {
        $file = '/images/favicons/apple-touch-icon.png';

        if (file_exists($this->site->getThemesPath($file))) {
            $this->links->add($this->site->getThemesUrl($file), array('rel'  => 'apple-touch-icon'));
        }
    }

    /**
     * humans.txt
     */
    protected function prepareHumansTxt()
    {
        $this->links->add(
            $this->site->getPermalink() . 'humans.txt',
            array(
                'rel'  => 'author',
                'type' => 'text/plain',
            )
        );
    }

    /**
     * sitemap.xml
     */
    protected function prepareSitemap()
    {
        $this->links->add(
            $this->site->getPermalink() . 'sitemap.xml',
            array(
                'rel'   => 'sitemap',
                'type'  => 'application/xml',
                'title' => 'Sitemap',
            )
        );
    }

    /**
     * Javascript in <head>
     */
    protected function prepareHeaderJavascript()
    {
        $this->js_header->addInline(file_get_contents(__DIR__ . '/../Resources/public/js/cookieManager.min.js'));
        $this->js_header->addInline(file_get_contents(__DIR__ . '/../Resources/public/js/firstUserInteraction.min.js'));
    }
}
