<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\LinkTypeDetails;
use Application\Core\Entity\LinkTypeInterface;
use Application\Core\LinkType\Binder;
use Application\Core\LinkType\ResourceInterface;
use Pongho\Core\Kernel;

/**
 * LinkType.
 *
 * @property int    $id
 * @property int    $language_id
 * @property string $link_type
 * @property int    $model_id
 * @property string $model_path
 * @property string|\Application\Admin\Form\LinkField\BaseHandler $handler
 */
class LinkType extends Base implements LinkTypeInterface
{
    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
    public static $table_name = 'link_types';

    /**
     * Callback 'after_save'.
     *
     * @var array
     * @static
     */
    public static $after_save = array('updateCache');

    /**
     * Trova il LinkType usando lingua, modello e tipo
     *
     * @static
     * @param integer $language_id
     * @param integer $model_id
     * @param string  $link_type
     * @return \Application\Core\Model\LinkType[]
     */
    public static function findByLanguageAndModelIdAndLinkType($language_id, $model_id, $link_type)
    {
        return self::all(
            array(
                'conditions' => array(
                    'language_id = :language_id AND model_id = :model AND link_type = :link_type',
                    'language_id' => $language_id,
                    'model'       => $model_id,
                    'link_type'   => $link_type
                ),
            )
        );
    }

    /**
     * Prepara gli attributi per il salvataggio del modello
     *
     * @static
     * @param array   $attributes
     * @param integer $site_id
     * @param integer $language_id
     * @return array
     *
     * @deprecated
     */
    public static function prepareLinkTypeAttributes(array $attributes, $language_id, $site_id)
    {
        $pieces = explode('::', $attributes['link_type']);
        $link_type = explode('_', $pieces[1]);

        $attributes['handler'] = $pieces[0];
        $attributes['site_id'] = $site_id;

        if (isset($link_type[1])) {
            $attributes['link_type'] = $link_type[0];
            $attributes['language_id'] = intval($link_type[1]);
            $attributes['key'] = $pieces[1];
        } else {
            $attributes['link_type'] = $link_type[0];
            $attributes['language_id'] = $language_id;
        }

        /** @var \Application\Admin\Form\LinkField\BaseHandler $handler */
        $handler = $attributes['handler'];

        return $handler::parseAttributes($attributes, $language_id);
    }

    /**
     * Richiama l'handler del LinkType per svuotare la cache corrispondente
     */
    public function updateCache()
    {
        /** @var Binder $binder */
        $binder = Kernel::instance()->getContainer()->get('link_type_binder');

        $binder->getBound($this->getIdentifier())->updateCache($this);
    }

    /**
     * Restituisce il permalink della risorsa linkata
     *
     * @return string
     */
    public function getPermalink()
    {
//        /** @var Binder $binder */
//        $binder = Kernel::instance()->getContainer()->get('link_type_binder');
//
//        return $binder->getBound($this->getIdentifier())->getPermalink($this);

        return absolute_url($this->getUrl());
    }

    /**
     * Restituisce la risorsa linkata dal LinkType
     *
     * @return ResourceInterface
     */
    public function getResource()
    {
        /** @var Binder $binder */
        $binder = Kernel::instance()->getContainer()->get('link_type_binder');

        return $binder->getBound($this->getIdentifier())->getResource($this);
    }

    /**
     * Restituisce il titolo della risorsa linkata, nella lingua corrente
     *
     * @return string
     */
    public function getTitle()
    {
        /** @var Binder $binder */
        $binder = Kernel::instance()->getContainer()->get('link_type_binder');

        return $binder->getBound($this->getIdentifier())->getTitle($this);
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getIdentifier()
    {
        return $this->link_type;
    }

    /**
     * {@inheritdoc}
     */
    public function getUrl()
    {
        return $this->model_path;
    }

    /**
     * {@inheritdoc}
     */
    public function getLinkDetails()
    {
        return new LinkTypeDetails($this->language_id, $this->model_id);
    }

    /**
     * {@inheritdoc}
     */
    public function change($identifier, $url = '', LinkTypeDetails $details)
    {
        $this->link_type = $identifier;
        $this->model_path = $url;
        $this->language_id = $details->getLanguageId();
        $this->model_id = $details->getModelId();

        return $this->save();
    }
}
