<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\AccountInterface;
use Application\Core\Entity\CountryInterface;
use Application\Core\Entity\ProvinceInterface;
use Application\Core\Entity\UserDetailsInterface;
use Application\Core\I18n\Address\Address;
use Application\Core\I18n\Country\Italy\AddressRenderer;
use Application\Core\Utilities\Validator;
use Pongho\Core\Kernel;
use Pongho\EventDispatcher\Event;
use Pongho\Utilities\DateTime;

/**
 * Modello per gli account.
 *
 * @property int                                $id
 * @property int                                $language_id
 * @property int                                $role_id
 * @property string                             $username
 * @property string                             $email
 * @property string                             $password
 * @property int                                $avatar_id
 * @property string                             $description
 * @property string                             $activation_key
 * @property string                             $reset_password_key
 * @property string                             $fullname
 * @property bool                               $is_active
 * @property bool                               $is_founder
 * @property bool                               $is_business
 * @property DateTime                           $created_at
 * @property DateTime                           $updated_at
 * @property DateTime                           $last_login_at
 * @property DateTime                           $expire_at
 * @property string                             $name
 * @property string                             $surname
 * @property string                             $company
 * @property string                             $partita_iva
 * @property string                             $codice_fiscale
 * @property string                             $address
 * @property string                             $address2
 * @property string                             $city
 * @property string                             $postcode
 * @property int                                $province_id
 * @property int                                $country_id
 * @property string                             $telephone
 * @property string                             $fax
 * @property string                             $website
 * @property string                             $googleplus
 * @property bool                               $newsletter
 * @property string                             $newsletter_activation_key
 * @property string                             $url
 * @property string                             $old_url
 * @property int                                $shipping_address_id
 *
 * @property int                                $total_orders
 *
 * @property \Application\Core\Model\Language   $language
 * @property \Application\Core\Model\Role       $role
 * @property \Application\Core\Model\Province   $province
 * @property \Application\Core\Model\Country    $country
 * @property \Application\Core\Model\Group[]    $groups
 * @property \Application\Core\Model\Address[]  $addresses
 * @property \Application\Core\Model\Address    $shipping_address
 */
class Account extends Base implements AccountInterface, UserDetailsInterface
{
    /**
     * Identificativo dell’utente anonimo.
     */
    const ANONYMOUS = 1;

    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
    public static $table_name = 'users';

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     * @static
     */
    public static $belongs_to = array(
        array('language', 'model' => 'Application\\Core\\Model\\Language'),
        array('role', 'model' => 'Application\\Core\\Model\\Role'),
        array('province', 'model' => 'Application\\Core\\Model\\Province'),
        array('country', 'model' => 'Application\\Core\\Model\\Country'),
        array('shipping_address', 'model' => 'Application\\Core\\Model\\Address'),
    );

    /**
     * Relazioni 'has_and_belongs_to_many'.
     *
     * @var array
     * @static
     */
    public static $has_and_belongs_to_many = array(
        array('groups', 'model' => 'Application\\Core\\Model\\Group'),
    );

    /**
     * Relazioni 'has_many'.
     *
     * @var array
     * @static
     */
    public static $has_many = array(
        array('addresses', 'model' => 'Application\\Core\\Model\\Address', 'foreign_key' => 'user_id'),
    );

    /**
     * Relazioni 'count_has_many'.
     *
     * @var array
     * @static
     */
    public static $count_has_many = array(
        array('total_orders', 'model' => 'Application\\Showcase\\Model\\Order', 'foreign_key' => 'customer_id'),
    );

    /**
     * Validazioni 'validates_presence_of'.
     *
     * @var array
     * @static
     */
    public static $validates_presence_of = array(
        array('username', 'message' => 'required_username'),
        array('email', 'message' => 'required_email'),
        array('password', 'message' => 'required_password', 'on' => 'create'),
    );

    /**
     * Validazioni 'validates_uniqueness_of'.
     *
     * @var array
     * @static
     */
    public static $validates_uniqueness_of = array(
        array('url', 'message' => 'invalid_url_uniqueness'),
    );

    /**
     * Validazioni 'validates_format_of'.
     *
     * @var array
     * @static
     */
    public static $validates_format_of = array(
        array('email', 'message' => 'invalid_email_format', 'with' => REGEXP_VALIDATE_EMAIL),
    );

    /**
     * Validazioni 'validates_by_methods_of'.
     *
     * @var array
     * @static
     */
    public static $validates_by_methods_of = array(
        array('username', 'method' => 'validate_username_format', 'message' => 'invalid_username_format'),
        array('username', 'method' => 'validate_uniqueness_of_username', 'message' => 'invalid_username_uniqueness'),
        array('email', 'method' => 'validate_uniqueness_of_email', 'message' => 'invalid_email_uniqueness'),
        array('partita_iva', 'method' => 'validate_partita_iva', 'message' => 'invalid_partita_iva_format'),
        array('codice_fiscale', 'method' => 'validate_codice_fiscale', 'message' => 'invalid_codice_fiscale_format'),
    );

    /**
     * Callback 'after_validation_on_create'.
     *
     * @var array
     * @static
     */
    public static $after_validation_on_create = array('validate_privacy', 'validate_confirm_password_on_create');

    /**
     * Callback 'after_validation_on_update'.
     *
     * @var array
     * @static
     */
    public static $after_validation_on_update = array('validate_confirm_password_on_update');

    /**
     * Callback 'after_validation'.
     *
     * @var array
     * @static
     */
    public static $after_validation = array('after_validation');

    /**
     * Callback 'before_create'.
     *
     * @var array
     * @static
     */
    public static $before_create = array('generateUrl');

    /**
     * Callback 'after_create'.
     *
     * @var array
     * @static
     */
    public static $after_create = array('addSelfPortfolioRelation');

    /**
     * Callback 'before_destroy'.
     *
     * @var array
     * @static
     */
    public static $before_destroy = array('delGroups', 'delete_newsletter_recipients');

    /**
     * Lista dei gruppi associati all'utente.
     *
     * @var \Application\Core\Model\Group[]
     */
    protected $groups;

    /**
     * Cache interna - Avatar.
     *
     * @var \Application\Core\Model\File
     */
    private $avatar;

    /**
     * Cerca un account in base all’indirizzo email.
     *
     * @param string $email
     * @return Account
     */
    public static function findByEmail($email)
    {
        if (!$email) {
            return null;
        }

        return static::first(
            array(
                'conditions' => array('email = ?', $email),
            )
        );
    }

    /**
     * Cerca un account in base allo username.
     *
     * @param string $username
     * @return Account
     */
    public static function findByUsername($username)
    {
        if (!$username) {
            return null;
        }

        return static::first(
            array(
                'conditions' => array('username = ?', $username),
            )
        );
    }

    /**
     * Cerca un account in base alla chiave di reset della password.
     *
     * @param string $key
     * @return Account
     */
    public static function findByResetPasswordKey($key)
    {
        if (!$key) {
            return null;
        }

        return static::first(
            array(
                'conditions' => array('reset_password_key = ?', $key),
            )
        );
    }

    /**
     * {@inheritdoc}
     */
    public function __set($name, $value)
    {
        // Caso speciale per la password.
        if ($name === 'password' || $name === 'confirm_password') {
            $value = clean_password($value);

            if ($name === 'password' && $value === '') {
                return;
            }
        }

        parent::__set($name, $value);
    }

    /**
     * Controlla che il checkbox sulla privacy sia spuntato.
     *
     * @ignore
     */
    public function validate_privacy()
    {
        if ($this->attributePresent('privacy') && $this->attributes['privacy']) {
            return;
        }

        $this->errors->add('privacy', 'required_privacy');
    }

    /**
     * Verifica la correttezza della conferma della password nel caso di nuovo account.
     *
     * È previsto che il campo `confirm_password` non sia presente, ma sia obbligatorio se presente. Questa ulteriore
     * verifica va fatta dal controller controllando i dati inviati dalla richiesta HTTP.
     *
     * @ignore
     */
    public function validate_confirm_password_on_create()
    {
        $this->validate_confirm_password();
    }

    /**
     * Verifica la correttezza della conferma della password in caso di modifica dell’account.
     *
     * A differenza della creazione di un nuovo account, la modifica prevede l'obbligo della conferma della password se
     * la password è stata modificata. Questa differenza perché in fase di creazione si suppone che il controller invii
     * una notifica via e-mail con i dati inseriti dall’utente, password compresa. Ma tale notifica non viene inviata
     * in caso di modifica.
     *
     * @ignore
     */
    public function validate_confirm_password_on_update()
    {
        if ($this->attributeEdited('password') && $this->password != '') {
            if ($this->attributePresent('confirm_password')) {
                $this->validate_confirm_password();
            } else {
                $this->errors->add('confirm_password', 'required_confirm_password');
            }
        }
    }

    /**
     * Verifica la correttezza della conferma della password se presente.
     *
     * @ignore
     */
    protected function validate_confirm_password()
    {
        if ($this->attributePresent('confirm_password')) {
            if (empty($this->confirm_password)) {
                $this->errors->add('confirm_password', 'required_confirm_password');
            } elseif ($this->password != $this->confirm_password) {
                $this->errors->add('confirm_password', 'password_mismatch');
            }
        }
    }

    /**
     * Verifica che lo username non sia uguale ad una e-mail diversa da quella dell’account.
     *
     * @param string $username
     * @return bool
     *
     * @ignore
     */
    public function validate_username_format($username)
    {
        if (preg_match(REGEXP_VALIDATE_EMAIL, $username)) {
            return $this->email === $username;
        }

        return true;
    }

    /**
     * Verifica l’unicità dello username e-mail.
     *
     * @param string $username
     * @return bool
     *
     * @ignore
     */
    public function validate_uniqueness_of_username($username)
    {
        if (preg_match(REGEXP_VALIDATE_EMAIL, $username)) {
            return true;
        }

        $conditions = array('username = :username', 'username' => $username);

        if (!$this->isNewRecord()) {
            $primary_keys = $this->primaryKey();

            $primary_key_conditions = array();
            foreach ($primary_keys as $pk_name) {
                $primary_key_conditions[] = "$pk_name = :$pk_name";
                $conditions[$pk_name] = $this->$pk_name;
            }

            $conditions[0] .= ' AND NOT ( ' . implode(' AND ', $primary_key_conditions) . ' )';
        }

        if (static::count(array('conditions' => $conditions))) {
            return false;
        }

        return true;
    }

    /**
     * Verifica l’unicità dell’indirizzo e-mail.
     *
     * @param string $email
     * @return bool
     *
     * @ignore
     */
    public function validate_uniqueness_of_email($email)
    {
        $conditions = array('email = :email', 'email' => $email);

        if (!$this->isNewRecord()) {
            $primary_keys = $this->primaryKey();

            $primary_key_conditions = array();
            foreach ($primary_keys as $pk_name) {
                $primary_key_conditions[] = "$pk_name = :$pk_name";
                $conditions[$pk_name] = $this->$pk_name;
            }

            $conditions[0] .= ' AND NOT ( ' . implode(' AND ', $primary_key_conditions) . ' )';
        }

        if (static::count(array('conditions' => $conditions))) {
            return false;
        }

        return true;
    }

    /**
     * Verifica la validità della partita iva. Il campo non è comunque obbligatorio.
     *
     * @param string $partita_iva
     * @return bool
     *
     * @ignore
     */
    public function validate_partita_iva($partita_iva)
    {
        return Validator::partita_iva($partita_iva);
    }

    /**
     * Verifica la validità del codice fiscale. Il campo non è comunque obbligatorio.
     *
     * @param string $codice_fiscale
     * @return bool
     *
     * @ignore
     */
    public function validate_codice_fiscale($codice_fiscale)
    {
        return Validator::codice_fiscale($codice_fiscale);
    }

    /**
     * Imposta il campo fullname.
     *
     * @ignore
     */
    public function after_validation()
    {
        $this->fullname = $this->name();
    }

    /**
     * Restituisce i gruppi ai quali fa parte l'utente.
     *
     * @return \Application\Core\Model\Group[]
     */
    public function getGroups()
    {
        if ($this->groups === null) {
            $this->groups = array();

            $groups = GroupUser::all(
                array(
                    'joins'      => 'LEFT JOIN ' . Group::tableName() . ' AS g ON g.id = `from`.group_id',
                    'conditions' => array('`from`.user_id = ?', $this->id)
                )
            );

            /** @var \Application\Core\Model\Group $group */
            foreach ($groups as $group) {
                $this->groups[$group->id] = $group;
            }
        }

        return $this->groups;
    }

    /**
     * Imposta i gruppi dell'utente.
     *
     * @param array $groups
     * @return $this
     */
    public function setGroups(array $groups)
    {
        // preferisco lavorare con gli ID dei gruppi
        if (reset($groups) instanceof Group) {
            foreach ($groups as &$group) {
                $group = $group->id;
            }
        }

        // preferisco operare eseguendo una transazione
        $account = $this;

        self::transaction(
            function () use ($account, $groups) {
                // rimuovo tutti i permessi
                $account->delGroups();

                foreach ($groups as $group_id) {
                    GroupUser::create(
                        array(
                            'group_id' => $group_id,
                            'user_id'  => $account->id,
                        )
                    );
                }
            }
        );

        $this->groups = null;

        return $this;
    }

    /**
     * Elimina le associazioni con i gruppi.
     *
     * @return $this
     */
    public function delGroups()
    {
        $groups = GroupUser::all(
            array(
                'conditions' => array('user_id = ?', $this->id)
            )
        );

        foreach ($groups as $group) {
            $group->delete();
        }

        return $this;
    }

    /**
     * Verifica se l’utente appartiene ad un gruppo.
     *
     * È possibile specificare anche una lista di gruppi passando come argomento un array. In questo caso, il metodo
     * restituisce true solo se l'account fa parte di tutti i gruppi passati.
     *
     * Il gruppo può essere sia istanza di Group che l'ID del gruppo.
     *
     * @param mixed $groups Può essere l’ID del gruppo o un array di ID. Può essere anche un modello gruppo, o un array di modelli.
     * @return $this
     */
    public function inGroup($groups)
    {
        if (!is_array($groups)) {
            $groups = array($groups);
        }

        foreach ($groups as &$group) {
            if ($group instanceof Group) {
                $group = $group->id;
            }
        }

        if (count($groups) == 1) {
            return array_key_exists(reset($groups), $this->getGroups());
        } else {
            $result = true;

            foreach ($groups as $group) {
                if (!array_key_exists($group, $this->getGroups())) {
                    $result = false;
                }
            }

            return $result;
        }
    }

    /**
     * Associa un gruppo all’utente.
     *
     * @param int|\Application\Core\Model\Group $group
     * @return $this
     */
    public function addGroup($group)
    {
        if ($group instanceof Group) {
            $group = $group->id;
        }

        $rel = GroupUser::find(array($group, $this->id));

        if ($rel === null) {
            GroupUser::create(
                array(
                    'group_id' => $group,
                    'user_id'  => $this->id,
                )
            );
        }

        return $this;
    }

    /**
     * Rimuove un gruppo dall’utente.
     *
     * @param int|\Application\Core\Model\Group|\Application\Core\Model\GroupUser $group
     * @return $this
     */
    public function delGroup($group)
    {
        if ($group instanceof GroupUser) {
            $group->delete();
        } else {
            if ($group instanceof Group) {
                $group = $group->id;
            }

            $rel = GroupUser::find(array($group, $this->id));

            if ($rel) {
                $rel->delete();
            }
        }

        return $this;
    }

    /**
     * Elenco dei genitori diretti, usato per cache interna.
     *
     * @static
     * @var \Application\Core\Model\Account[]
     */
    protected static $parents;

    /**
     * Restituisce i genitori diretti dell’utente.
     *
     * @return \Application\Core\Model\Account[]
     *
     * @api
     */
    public function getParents()
    {
        if (self::$parents === null) {
            $options = array(
                'select'     => '`from`.*',
                'joins'      => 'LEFT JOIN ' . AccountRelation::tableName() . ' AS r ON r.parent_id = `from`.id',
                'conditions' => array('child_id = ?', $this->id),
            );

            self::$parents = static::all($options);
        }

        return self::$parents;
    }

    /**
     * Imposta i genitori diretti dell’utente.
     *
     * @param \Application\Core\Model\Account[] $parents
     * @return $this
     *
     * @api
     */
    public function setParents(array $parents)
    {
        $old_parents = array();
        foreach ($this->getParents() as $old_parent) {
            $old_parents[$old_parent->id] = $old_parent;
        }

        foreach ($parents as $parent_id_or_parent_instance) {
            if ($parent_id_or_parent_instance instanceof static) {
                $parent_id = $parent_id_or_parent_instance->id;
            } else {
                $parent_id = $parent_id_or_parent_instance;
            }

            if (isset($old_parents[$parent_id])) {
                unset($old_parents[$parent_id]);
            } else {
                $this->setParent($parent_id);
            }
        }

        foreach ($old_parents as $old_parent) {
            if (!$this->delParent($old_parent)) {
                return false;
            }
        }

        self::$parents = null;

        return $this;
    }

    /**
     * Imposta un genitore diretto dell’utente.
     *
     * @param int|\Application\Core\Model\Account $parent_id_or_parent_instance
     * @throws \LogicException
     * @return $this
     *
     * @api
     */
    public function setParent($parent_id_or_parent_instance)
    {
        if ($parent_id_or_parent_instance instanceof static) {
            $parent_id = $parent_id_or_parent_instance->id;
        } else {
            $parent_id = $parent_id_or_parent_instance;
        }

        if ($parent_id === $this->id) {
            throw new \LogicException('The user cannot be a parent or a child of himself');
        }

        if (!$this->hasParent($parent_id)) {
            // Creo una relazione padre-figlio
            /** @var \Application\Core\Model\AccountRelation $relation */
            $relation = AccountRelation::create(
                array(
                    'parent_id' => $parent_id,
                    'child_id'  => $this->id,
                )
            );

            // Creo le relazioni antenato-figlio. Gli antenati del figlio sono il padre e gli antenati del padre.
            $ancestors_options = array(
                'conditions' => array('descendant_id = ?', $parent_id),
            );

            /** @var \Application\Core\Model\AccountPortfolio $ancestor_relation */
            foreach (AccountPortfolio::all($ancestors_options) as $ancestor_relation) {
                $attributes = array(
                    'relation_id'   => $relation->id,
                    'ancestor_id'   => $ancestor_relation->ancestor_id,
                    'descendant_id' => $this->id,
                    'depth'         => $ancestor_relation->depth + 1,
                );

                AccountPortfolio::create($attributes);
            }
        }

        self::$parents = null;

        return $this;
    }

    /**
     * Rimuove un genitore diretto dell’utente.
     *
     * @param int|\Application\Core\Model\Account $parent_id_or_instance
     * @return bool
     *
     * @api
     */
    public function delParent($parent_id_or_instance)
    {
        if ($parent_id_or_instance instanceof static) {
            $parent_id = $parent_id_or_instance->id;
        } else {
            $parent_id = $parent_id_or_instance;
        }

        /** @var \Application\Core\Model\AccountRelation $relation */
        $relation = AccountRelation::first(
            array(
                'conditions' => array('parent_id = ? AND child_id = ?', $parent_id, $this->id),
            )
        );

        if ($relation !== null) {
            $ancestors_options = array(
                'conditions' => array('relation_id = ?', $relation->id),
            );

            /** @var \Application\Core\Model\AccountPortfolio $ancestor_relation */
            foreach (AccountPortfolio::all($ancestors_options) as $ancestor_relation) {
                if (!$ancestor_relation->delete()) {
                    return false;
                }
            }

            return $relation->delete();
        }

        self::$parents = null;

        return true;
    }

    /**
     * Indica se l’utente ha un genitore diretto.
     *
     * @param int $parent_id L'ID del genitore che si vuole controllare.
     * @return bool
     *
     * @api
     */
    public function hasParent($parent_id)
    {
        $options = array(
            'conditions' => array('parent_id = ? AND child_id = ?', $parent_id, $this->id),
        );

        return !(AccountRelation::find($options) === null);
    }

    /**
     * Indica se l'utente ha un certo antenato
     *
     * @param $ancestor_id
     * @return bool
     */
    public function hasAncestor($ancestor_id)
    {
        $options = array(
            'conditions' => array(
                'ancestor_id = :ancestor AND descendant_id = :descendant',
                'ancestor'   => $ancestor_id,
                'descendant' => $this->id,
            ),
        );

        return !(AccountPortfolio::find($options) === null);
    }

    /**
     * Restituisce i discendenti.
     *
     * @return \ActiveRecord\Collection|Account[]
     */
    public function getDescendants()
    {
        $options = array(
            'joins' => 'INNER JOIN ' . AccountPortfolio::tableName() . ' AS p ON p.descendant_id = `from`.id',
            'conditions' => array(
                'ancestor_id = :ancestor AND descendant_id != :me',
                'ancestor' => $this->id,
                'me'       => $this->id,
            ),
            'collection' => true,
        );

        return Account::all($options);
    }

    /**
     * Restituisce il numero di discendenti.
     *
     * @return int
     */
    public function countDescendants()
    {
        $options = array(
            'conditions' => array(
                'ancestor_id = :ancestor AND descendant_id != :me',
                'ancestor' => $this->id,
                'me'       => $this->id,
            ),
        );

        return AccountPortfolio::count($options);
    }

    /**
     * Indica se l'utente ha un certo discendente
     *
     * @param $descendant_id
     * @return bool
     */
    public function hasDescendant($descendant_id)
    {
        $options = array(
            'conditions' => array(
                'ancestor_id = :ancestor AND descendant_id = :descendant',
                'ancestor'   => $this->id,
                'descendant' => $descendant_id,
            ),
        );

        return !(AccountPortfolio::find($options) === null);
    }

    /**
     * Genera l'url alla creazione dell'utente.
     */
    public function generateUrl()
    {
        $date = new DateTime();

        $timestamp = $date->format('U');

        $this->url = (string)$timestamp . (string)random(8, '0123456789');
    }

    /**
     * @return string
     */
    public function getUrl()
    {
        return $this->url;
    }

    /**
     * Aggiunge la relazione a se stesso nella tabella del portfolio.
     *
     * @ignore
     */
    public function addSelfPortfolioRelation()
    {
        $attributes = array(
            'relation_id'   => null,
            'ancestor_id'   => $this->id,
            'descendant_id' => $this->id,
            'depth'         => 0,
        );

        AccountPortfolio::create($attributes);
    }

    /**
     * Elimina le relazioni dell’utente e sposta tutti i suoi figli sotto il controllo di tutti i suoi padri.
     *
     * @return bool
     *
     * @ignore
     */
    public function deletePortfolioRelations()
    {
        // Ricavo l’elenco degli ID dei genitori
        $parents_options = array(
            'conditions' => array('child_id = ?', $this->id),
        );

        $parents = array();
        /** @var \Application\Core\Model\AccountRelation $relation */
        foreach (AccountRelation::all($parents_options) as $relation) {
            $parents[] = $relation->parent_id;
        }

        // Per ogni Account figlio, associo i nuovi padri
        $children_options = array(
            'select'     => '`from`.*',
            'joins'      => 'LEFT JOIN ' . AccountRelation::tableName() . ' AS ar ON ar.child_id = `from`.id',
            'conditions' => array('ar.parent_id = ?', $this->id),
        );

        /** @var \Application\Core\Model\Account $children */
        foreach (Account::all($children_options) as $children) {
            foreach ($parents as $parent_id) {
                $children->setParent($parent_id);
            }
        }

        // A questo punto, elimino tutte le associazioni dell’utente
        $portfolio_options = array(
            'conditions' => array('ancestor_id = :account OR descendant_id = :account', 'account' => $this->id),
        );

        /** @var \Application\Core\Model\AccountPortfolio $portfolio */
        foreach (AccountPortfolio::all($portfolio_options) as $portfolio) {
            $portfolio->delete();
        }

        $relations_options = array(
            'conditions' => array('parent_id = :account OR child_id = :account', 'account' => $this->id),
        );

        /** @var \Application\Core\Model\AccountRelation $relation */
        foreach (AccountRelation::all($relations_options) as $relation) {
            $relation->delete();
        }

        return true;
    }

    /**
     * @todo
     * @ignore
     */
    public function delete_newsletter_recipients()
    {
//		$rows = NewsletterSend::all(array(
//			'conditions'	=> array('recipient_id = ?', $this->id)
//		));
//
//		foreach ( $rows as $row )
//		{
//			$row->recipient_id = null;
//			$row->save();
//		}
    }

    /**
     * {@inheritdoc}
     */
    public function isActive()
    {
        return $this->is_active;
    }

    /**
     * Indica se l’utente è anonimo.
     *
     * @return boolean
     */
    public function isAnonymous()
    {
        return (boolean)($this->id == self::ANONYMOUS);
    }

    /**
     * Indica se l’utente è amministratore.
     *
     * @return boolean
     */
    public function isAdmin()
    {
        return (boolean)($this->role_id == Role::ADMIN);
    }

    /**
     * Indica se l’utente è founder.
     *
     * @return boolean
     */
    public function isFounder()
    {
        return (boolean)$this->is_founder;
    }

    /**
     * Indica se l’utente è un cliente business.
     *
     * @return boolean
     */
    public function isBusiness()
    {
        return (boolean)$this->is_business;
    }

    /**
     * Indica se l’utente ha il permesso richiesto.
     *
     * @param string $key Chiave del permesso.
     * @return boolean
     */
    public function hasPermit($key)
    {
        // Il founder ha automaticamente qualsiasi permesso
        if ($this->isFounder()) {
            return true;
        }

        if ($role = $this->role) {
            return $role->hasPermit($key);
        }

        return false;
    }

    /**
     * È possibile modificare il comportamento di questa funzione tramite l'evento model.pongho_users.name_function.
     *
     * {@inheritdoc}
     */
    public function name($invert = false)
    {
        $event = new Event($this, $this->getEventName('name_function'), array('invert' => $invert));

        if ($name = $this->dispatcher->notifyUntil($event)->getReturnValue()) {
            return $name;
        }

        if ($invert) {
            $name = trim("$this->surname $this->name");
        } else {
            $name = trim("$this->name $this->surname");
        }

        return empty($name) ? $this->username : $name;
    }

    /**
     * {@inheritdoc}
     */
    public function getActivationKey()
    {
        return $this->activation_key;
    }

    /**
     * {@inheritdoc}
     */
    public function resetActivationKey()
    {
        $this->activation_key = random(32);

        return $this;
    }

    /**
     * Elimina l’account.
     *
     * @return boolean
     */
    public function delete()
    {
        if (!$this->isDeletable()) {
            return false;
        }

        try {
            $event = new Event($this, $this->getEventName('before_delete'));
            $this->dispatcher->notify($event);
        } catch (\Exception $e) {
            return false;
        }

        if (!$this->deleteGroupsRelations()) {
            return false;
        }

        if (!$this->deleteAvatar()) {
            return false;
        }

        if (!$this->deletePortfolioRelations()) {
            return false;
        }

        if (!parent::delete()) {
            return false;
        }

        try {
            $event = new Event($this, $this->getEventName('after_delete'));
            $this->dispatcher->notify($event);
        } catch (\Exception $e) {
            return false;
        }

        return true;
    }

    /**
     * isDeletable
     *
     * @return bool
     */
    public function isDeletable()
    {
        $event = new Event($this, $this->getEventName('is_deletable'), array('is_deletable' => true));
        $this->dispatcher->notify($event);

        return $event->getParameter('is_deletable');
    }

    /**
     * Reimposta l'autore di nodi e commenti ad anonimo prima di cancellare l'utente che li ha creati.
     *
     * @return boolean
     *
     * @deprecated
     */
    public function updateCmsRelations()
    {
        /* @var \Application\Core\Model\Manager\ApplicationManager $app_manager */
        $app_manager = Kernel::instance()->getContainer()->get('application_manager');

        if ($app_manager->isInstalled('Cms')) {
            $options = array('conditions' => array('author_id = :user', 'user' => $this->id));

            /** @var \ActiveRecord\Base $node_model_class */
            $node_model_class = 'Application\\Cms\\Model\\Node';

            foreach ($node_model_class::all($options) as $node) {
                $node->author_id = 1;
                if (!$node->save()) {
                    return false;
                }
            }

            /** @var \ActiveRecord\Base $comment_model_class */
            $comment_model_class = 'Application\\Cms\\Model\\Comment';

            foreach ($comment_model_class::all($options) as $comment) {
                $comment->author_id = 1;
                $comment->author_name = $this->name();
                $comment->author_email = $this->email;

                if (!$comment->save()) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Rimuove l’utente dai gruppi.
     *
     * @return boolean
     */
    public function deleteGroupsRelations()
    {
        $options = array('conditions' => array('user_id = :user', 'user' => $this->id));

        foreach (GroupUser::all($options) as $group) {
            if (!$group->delete()) {
                return false;
            }
        }

        return true;
    }

    /**
     * Rimuovo i reminder dell’utente
     *
     * @return boolean
     *
     * @deprecated
     */
    public function deleteReminders()
    {
        /* @var \Application\Core\Model\Manager\ApplicationManager $app_manager */
        $app_manager = Kernel::instance()->getContainer()->get('application_manager');

        if ($app_manager->isInstalled('Reminder')) {
            $options = array('conditions' => array('user_id = :user', 'user' => $this->id));

            /** @var \ActiveRecord\Base $reminder_model_class */
            $reminder_model_class = 'Application\\Reminder\\Model\\Reminder';

            foreach ($reminder_model_class::all($options) as $reminder) {
                if (!$reminder->delete()) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Elimina l’avatar utente.
     *
     * @return boolean
     */
    public function deleteAvatar()
    {
        if ($this->avatar_id) {
            $image = File::find($this->avatar_id);

            $this->avatar_id = null;
            $this->save(false);

            return $image->delete();
        }

        return true;
    }

    /**
     * @return int
     */
    public function getAvatarId()
    {
        return $this->avatar_id;
    }

    /**
     * Restituisce il modello dell’avatar.
     *
     * @return File
     */
    public function getAvatar()
    {
        if ($this->avatar === null && $this->avatar_id) {
            $this->avatar = File::find($this->avatar_id);
        }

        return $this->avatar;
    }

    /**
     * Alias di getAvatar().
     *
     * @see Account::getAvatar() Metodo di riferimento.
     *
     * @return File
     */
    public function avatar()
    {
        return $this->getAvatar();
    }

    /**
     * Restituisce l’indirizzo nel formato indicato dal paese di appartenenza.
     *
     * @return string
     */
    public function renderAddress()
    {
        $address = new Address(
            $this->name,
            $this->surname,
            $this->company,
            $this->address,
            $this->address2,
            $this->postcode,
            $this->city,
            $this->province,
            $this->country,
            $this->telephone
        );

        // @todo AddressRenderer dovrebbe essere creato dal modello Country
        $renderer = new AddressRenderer();

        return $renderer->render($address);
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguage()
    {
        return $this->language;
    }

    /**
     * @return int
     */
    public function getLanguageId()
    {
        return $this->language_id;
    }

    /**
     * @param int|Language $language
     * @return $this
     */
    public function setLanguage($language)
    {
        if ($language instanceof Language) {
            $language_id = $language->id;
        } elseif (is_numeric($language)) {
            $language_id = (int) $language;
        } else {
            throw new \InvalidArgumentException('Expected an integer or an instance of Language.');
        }

        $this->language_id = $language_id;
        unset($this->language);

        return $this;
    }

    /**
     * @return Role
     */
    public function getRole()
    {
        return $this->role;
    }

    /**
     * @return int
     */
    public function getRoleId()
    {
        return $this->role_id;
    }

    /**
     * @param int|Role $role
     * @return $this
     */
    public function setRole($role)
    {
        if ($role instanceof Role) {
            $role_id = $role->id;
        } elseif (is_numeric($role)) {
            $role_id = (int) $role;
        } else {
            throw new \InvalidArgumentException('Expected an integer or an instance of Role.');
        }

        $this->role_id = $role_id;
        unset($this->role);

        return $this;
    }

    /**
     * @return string
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @throw \InvalidArgumentException
     * @param string $username
     */
    public function setUsername($username)
    {
        $this->username = $username;
    }

    /**
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @throw \InvalidArgumentException
     * @param string $email
     */
    public function setEmail($email)
    {
        $this->email = $email;
    }

    /**
     * Restituisce la versione criptata della password
     *
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @throw \InvalidArgumentException
     * @param string $password
     * @return $this
     */
    public function setPassword($password)
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @param string $password
     * @return $this
     */
    public function setConfirmPassword($password)
    {
        $this->confirm_password = $password;

        return $this;
    }

    /**
     * @return DateTime
     */
    public function getSubscribeDate()
    {
        return $this->created_at;
    }

    /**
     * @return DateTime
     */
    public function getCreationDate()
    {
        return $this->created_at;
    }

    /**
     * @return DateTime
     */
    public function getLastUpdateDate()
    {
        return $this->updated_at;
    }

    /**
     * @return DateTime
     */
    public function getLastLoginDate()
    {
        return $this->last_login_at;
    }

    /**
     * @return DateTime
     */
    public function getExpirationDate()
    {
        return $this->expire_at;
    }

    /**
     * @param DateTime $expiration_date
     * @return $this
     */
    public function setExpirationDate(DateTime $expiration_date)
    {
        $this->expire_at = $expiration_date;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function activate()
    {
        $this->is_active = true;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function deactivate()
    {
        $this->is_active = false;

        return $this;
    }

    /**
     * @return int|null
     */
    public function getUserId()
    {
        return $this->id;
    }

    /**
     * @return AccountInterface
     */
    public function getAccount()
    {
        return $this;
    }

    /**
     * @return UserDetailsInterface
     */
    public function getUserDetails()
    {
        return $this;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getSurname()
    {
        return $this->surname;
    }

    /**
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * @return string
     */
    public function getCompany()
    {
        return $this->company;
    }

    /**
     * @return string
     */
    public function getTaxCode()
    {
        return $this->codice_fiscale;
    }

    /**
     * @return string
     */
    public function getVatNumber()
    {
        return $this->partita_iva;
    }

    /**
     * @return string
     */
    public function getAddress()
    {
        return $this->address;
    }

    /**
     * @return string
     */
    public function getAddress2()
    {
        return $this->address2;
    }

    /**
     * @return string
     */
    public function getCity()
    {
        return $this->city;
    }

    /**
     * @return string
     */
    public function getPostCode()
    {
        return $this->postcode;
    }

    /**
     * @return ProvinceInterface
     */
    public function getProvince()
    {
        return $this->province;
    }

    /**
     * @return int
     */
    public function getProvinceId()
    {
        return $this->province_id;
    }

    /**
     * @return CountryInterface
     */
    public function getCountry()
    {
        return $this->country;
    }

    /**
     * @return int
     */
    public function getCountryId()
    {
        return $this->country_id;
    }

    /**
     * @return string
     */
    public function getTelephone()
    {
        return $this->telephone;
    }

    /**
     * @return string
     */
    public function getMobile()
    {
        throw new \LogicException('Not implemented');
    }

    /**
     * @return string
     */
    public function getFax()
    {
        return $this->fax;
    }

    /**
     * @return string
     */
    public function getWebsite()
    {
        return $this->website;
    }

    /**
     * @param $privacy
     * @return $this
     */
    public function setPrivacy($privacy)
    {
        $this->privacy = (bool) $privacy;

        return $this;
    }
}
