<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Entity\SiteInterface;
use Application\Core\I18n\Translation\Translator;
use Application\Core\Model\Manager\LanguageManager;
use Application\Core\Notification\ForgotPasswordNotification;
use Application\Core\Notification\NotificationCollection;
use Application\Core\Notification\NotificationSender;
use Application\Core\Notification\NotificationTemplateGenerator;
use Application\Core\Notification\SendActivationLinkNotification;
use Application\Core\Notification\SendCredentialsNotification;
use Application\Core\Notification\UserEditNotification;
use Application\Core\Notification\UserStatusActiveNotification;
use Application\Core\Notification\UserStatusNotActiveNotification;
use Application\Core\Notification\UserCreationActivationNotification;
use Application\Core\Notification\UserCreationNotification;
use Application\Core\Notification\UserSubscribeEmailNotification;
use Application\Core\Notification\UserSubscribeSimpleNotification;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class NotificationServiceProvider
 */
class NotificationServiceProvider implements ServiceProviderInterface
{
    /**
     * Permette di registrare dei servizi aggiuntivi al contenitore che gli viene passato.
     *
     * Questo metodo dovrebbe solo impostare o estendere dei servizi,
     * ma non dovrebbe mai richiederli.
     *
     * @param Container $container
     */
    public function register(Container $container)
    {
        $container->share('notification_collection', array($this, 'getNotificationCollection'));
        $container->set('notification_sender', array($this, 'getNotificationSender'));
    }

    /**
     * @param Container $container
     * @return NotificationCollection
     *
     * @internal
     */
    public function getNotificationCollection(Container $container)
    {
        $collection = new NotificationCollection();

        /** @var SiteInterface $site */
        $site = $container->get('site');

        /** @var Translator $trans */
        $trans = $container->get('translator');

        /** @var LanguageManager $language_manager */
        $language_manager = $container->get('language_manager');
        $langs = $language_manager->findAllBySite($site->getId());

        // Ogni notifica ha bisogno di una nuova istanza del generatore per gestire le diverse opzioni di configurazione
        // anche se non capita di usare più notifiche contemporaneamente, altrimenti c'è il rischio che una notifica possa
        // modificare le impostazioni di un'altra
        $collection
            ->add(new UserSubscribeSimpleNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new UserSubscribeEmailNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new ForgotPasswordNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new UserCreationNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new UserCreationActivationNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new UserEditNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new UserStatusActiveNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new UserStatusNotActiveNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new SendCredentialsNotification($site, $trans, new NotificationTemplateGenerator($langs)))
            ->add(new SendActivationLinkNotification($site, $trans, new NotificationTemplateGenerator($langs)))
        ;

        return $collection;
    }

    /**
     * @param Container $container
     * @return NotificationSender
     *
     * @internal
     */
    public function getNotificationSender(Container $container)
    {
        return new NotificationSender(
            $container->get('site'),
            $container->get('mailer'),
            $container->get('mailer_helper'),
            $container->get('notification_template_manager'),
            $container->get('logger')
        );
    }
}
