<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use ActiveRecord\CacheWrapper;
use ActiveRecord\Config;
use ActiveRecord\Connection;
use ActiveRecord\Profiler\Profiler;
use Doctrine\DBAL\Configuration;
use Doctrine\DBAL\DriverManager;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class DbServiceProvider
 */
class DbServiceProvider implements ServiceProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->share('db_profiler', array($this, 'getDbProfilerService'));
        $container->share('connection', array($this, 'getConnectionService'));
        $container->share('connection_schema', array($this, 'getConnectionSchemaService'));

        // Non sarebbe affar suo ma è per tenere tutto quello relativo al database nello stesso posto
        $this->registerOrm($container);
    }


    /**
     * Restituisce il profilatore delle query in database.
     *
     * @return \ActiveRecord\Profiler\ProfilerInterface
     *
     * @internal
     */
    public function getDbProfilerService()
    {
        return new Profiler();
    }

    /**
     * Restituisce la connessione (DBAL) al database.
     *
     * @return \Doctrine\DBAL\Connection
     *
     * @internal
     */
    public function getConnectionService()
    {
        $pdo = Connection::instance()->connection;
        $config = new Configuration();

        $connectionParams = array(
            'pdo' => $pdo,
        );

        return DriverManager::getConnection($connectionParams, $config);
    }

    /**
     * Restituisce lo schema (DBAL) del database.
     *
     * @param Container $container
     * @return \Doctrine\DBAL\Schema\Schema
     *
     * @internal
     */
    public function getConnectionSchemaService(Container $container)
    {
        /** @var \Doctrine\DBAL\Connection $conn */
        $conn = $container->get('connection');
        $sm = $conn->getSchemaManager();

        return $sm->createSchema();
    }

    /**
     * Imposta la configurazione per la connessione al database.
     *
     * @param Container $container
     */
    protected function registerOrm(Container $container)
    {
        $path = $container->getParameter('orm.config_path');

        if (!file_exists($path)) {
            throw new \UnexpectedValueException(sprintf('Database configuration file not exists on path "%s"!', $path));
        }

        $settings = include $path;

        if (!array_key_exists('connections', $settings)) {
            throw new \UnexpectedValueException(sprintf('Connections not found on %s file.', $path));
        }

        if (!array_key_exists('default_connection', $settings)) {
            throw new \UnexpectedValueException(sprintf('Default connection not found on %s file.', $path));
        }

        $default_connection = $settings['default_connection'];

        if ($container->get('kernel')->getEnvironment() === 'test' && isset($settings['connections']['tests'])) {
            $default_connection = 'tests';
        }

        Config::instance()
            ->setConnections($settings['connections'])
            ->setDefaultConnectionName($default_connection)
            ->setDebug($container->get('kernel')->isDebug())
            ->setEventDispatcher($container->get('event_dispatcher'));

        if ($container->hasService('db_profiler') && $container->get('kernel')->isDebug()) {
            Config::instance()->setProfiler($container->get('db_profiler'));
        }

        try {
            Config::instance()->setCache(new CacheWrapper($container->get('cache')));
        } catch (\BadMethodCallException $e) {
            // do nothing
        }
    }
}
