<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Console\Command\CacheClearCommand;
use Pongho\Caching\ApcCache;
use Pongho\Caching\FileCache;
use Pongho\Caching\XcacheCache;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

class CachingServiceProvider implements ServiceProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->share('cache', array($this, 'getCacheService'));

        $container->extend(
            'console_commands',
            function (\ArrayObject $commands, Container $container) {
                $commands->append(new CacheClearCommand($container->get('cache')));
            }
        );
    }

    /**
     * Restituisce il gestore della cache.
     *
     * @param Container $container
     * @return \Application\Core\Utilities\Shortcode
     *
     * @internal
     */
    public function getCacheService(Container $container)
    {
        if (!$container->getParameter('cache.enabled')) {
            throw new \BadMethodCallException('Cache disabled!');
        }

        $class = $container->getParameter('cache.cache_class');

        if ($class === null || !class_exists($class)) {
            if (function_exists('apc_exists')) {
                return new ApcCache('pongho');
            } elseif (function_exists('xcache_isset')) {
                return new XcacheCache('pongho');
            } else {
                return new FileCache('pongho', CACHE_PATH);
            }
        } else {
            return new $class();
        }
    }
}
