<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Controller;

use Application\Core\BaseApiController;
use Application\Core\Model\Account;
use Application\Core\Model\Province;
use Application\Core\Model\Role;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Template\Html;

/**
 * Class ApiController
 */
class ApiController extends BaseApiController
{
    /**
     * Ricava i valori inseriti nella tabella degli utenti
     *
     * <code>Il nome del comando intende che tipo di valore viene restituito, ad esempio:
     * - autocomplete/user_id?term=metaline restituirà un array con gli id degli utenti che contengono "metaline" nel nome, cognome o email
     * - autocomplete/username?term=meta restituirà gli username degli utenti che contengono "meta" nello username</code>
     *
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return array
     */
    public function autocompleteAction()
    {
        if ($response = $this->checkLogIn()) {
            return $response;
        }

        $command = $this->getParameter('command');

        $term = $this->getRequest()->query->get('term', '');
        $results = array();

        switch ($command) {
            case 'user_id':
                $options = array(
                    'conditions' => array(
                        'is_active = :active AND role_id <> :anon AND
                            (pongho_like(fullname, :term) OR pongho_like(name, :term) OR pongho_like(surname, :term) OR pongho_like(email, :term))',
                        'active'  => true,
                        'anon'    => Role::USER_NOT_LOGGED,
                        'term'    => '%' . trim($term) . '%',
                    ),
                );

                $notin = array();
                if (isset($_GET['not_in']) && is_array($_GET['not_in'])) {
                    foreach ($_GET['not_in'] as $id) {
                        $notin[] = intval($id);
                    }
                }

                if (!empty($notin)) {
                    $condition = 'id NOT IN (' . implode(', ', $notin) . ')';
                    $options = Account::addCondition($options, array($condition));
                }

                /** @var \Application\Core\Model\Account $user */
                foreach (Account::all($options) as $user) {
                    $results[] = array(
                        'id' => $user->id,
                        'label' => $user->name()
                    );
                }

                break;

            case 'username':
                $options = array(
                    'select' => 'username',
                    'conditions' => array('pongho_like(username, :term) AND id != 1', 'term' => '%' . $term . '%'),
                    'group' => 'username',
                    'order' => 'username ASC',
                );

                foreach (Account::all($options) as $account) {
                    $results[] = $account->username;
                }

                break;

            case 'city':
                /** @var \Doctrine\Dbal\Connection $connection */
                $connection = $this->getContainer()->get('connection');

                $sql = "SELECT city FROM (SELECT REGEXP_REPLACE(city, '\s+$', '') AS city FROM pongho_users WHERE pongho_like(TRIM(city), :term)) AS t GROUP BY city";
                $statement = $connection->prepare($sql);
                $statement->execute(array('term' => '%' . $term . '%'));

                foreach ($statement as $row) {
                    $results[] = $row['city'];
                }

                break;

            default:
                throw new HttpNotFoundException();
        }

        return $results;
    }

    /**
     * @return mixed
     */
    public function provinceAction()
    {
        if ($response = $this->checkLogIn()) {
            return $response;
        }

        $command = $this->getParameter('command');

        $provinces = array();
        if (strtolower($command) === 'all') {
            $options = array();

            // Filtro su regione
            if ($region_id = intval($this->getRequest()->query->get('r'))) {
                $options = Province::addCondition($options, array('region_id = :region', 'region' => $region_id));
            }

            // Ordinamento
            if ($order = $this->getRequest()->query->get('o')) {
                if (array_key_exists($order, Province::columns())) {
                    $options['order'] = $order;
                }
            }

            /** @var Province $province */
            foreach (Province::all($options) as $province) {
                $provinces[$province->id] = $province->name;
            }
        }

        // Formato dei dati restituito
        switch ($this->getRequest()->query->get('format')) {
            case 'html':
                /** @var Html $html */
                $html = $this->getContainer()->get('template_html');

                $provinces = $html->select($provinces);
                break;

            case 'json':
                $json_provinces = array();

                foreach ($provinces as $id => $province) {
                    $json_provinces[] = array(
                        'id' => $id,
                        'name' => $province,
                    );
                }

                $provinces = $json_provinces;
                break;

            default:
                break;
        }

        return $provinces;
    }
}
