<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele Termini
 * @package Application\Core
 */

namespace Application\Core\Utilities;

use Application\Core\Firewall;
use Application\Core\I18n\Translator\Translator;
use Pongho\Core\ControllerHelper as Base;
use Pongho\Core\TemplateResponse;
use Pongho\Http\JsonResponse;
use Pongho\Http\RedirectResponse;
use Pongho\Template\View;

/**
 * Class ControllerHelper
 */
class ControllerHelper extends Base
{
    /**
     * @var \Application\Core\Model\ModuleSite
     */
    protected $module_site;

    /**
     * Restituisce l’ID del sito corrente.
     *
     * @return integer
     */
    public function getSiteId()
    {
        return $this->container->getService('site')->id;
    }

    /**
     * @return string
     */
    public function getPath()
    {
        $path = trim($this->container->getParameter('path'), '/');

        return empty($path) ? '/' : '/' . $path . '/';
    }

    /**
     * Restituisce la testata.
     *
     * @return \Pongho\Template\HeadHelper
     */
    public function getHead()
    {
        return $this->container->getService('template_head');
    }

    /**
     * Restituisce la vista.
     *
     * @return \Pongho\Template\Theme
     */
    public function getTheme()
    {
        static $view;

        if ($view === null) {
            $view = $this->filter($this->container->getService('controller'), 'core.filter_view', $this->container->getService('theme_view'));

            $view->assignVars(
                array(
                    'module' => $this->getModuleSite(false),
                )
            );
        }

        return $view;
    }

    /**
     * Restituisce la vista.
     *
     * @access public
     * @return \Pongho\Template\Theme
     * @deprecated
     */
    public function getView()
    {
        return $this->getTheme();
    }

    /**
     * @param $template_path
     * @return View
     */
    public function createView($template_path)
    {
        $view = new View($template_path);
        $view->assignVars($this->container->getService('global_view_vars'));

        return $view;
    }

    /**
     * Visualizza un messaggio.
     *
     * @param string  $message
     * @param boolean $is_error
     * @param array   $parameters
     * @return \Pongho\Core\TemplateResponse|\Pongho\Http\JsonResponse
     */
    public function displayMessage($message, $is_error = false, array $parameters = array())
    {
        if ($this->container->getService('request')->isAjax()) {
            return $this->displayJsonMessage($message, $is_error, $parameters);
        }

        $parameters['message'] = $message;
        $parameters['error'] = $is_error;

        $this->getTheme()
            ->setTemplate('message.php')
            ->assignVars($parameters);

        return new TemplateResponse($this->getTheme(), $this->getResponseHeaders());
    }

    /**
     * Visualizza un messaggio in formato JSON.
     *
     * @param string  $message
     * @param boolean $is_error
     * @param array   $parameters
     * @return \Pongho\Http\JsonResponse
     */
    public function displayJsonMessage($message, $is_error = false, array $parameters = array())
    {
        $parameters['message'] = $message;
        $parameters['error'] = $is_error;

        return new JsonResponse($parameters);
    }

    /**
     * Visualizza un messaggio di errore.
     *
     * @param string $message
     * @param array  $parameters
     * @return \Pongho\Http\JsonResponse
     */
    public function displayError($message, array $parameters = array())
    {
        return $this->displayMessage($message, true, $parameters);
    }

    /**
     * Visualizza un messaggio di errore in formato JSON.
     *
     * @param string $message
     * @param array  $parameters
     * @return \Pongho\Http\JsonResponse
     */
    public function displayJsonError($message, array $parameters = array())
    {
        return $this->displayJsonMessage($message, true, $parameters);
    }

    /**
     * Restituisce una risposta per un redirect all’url passato.
     *
     * Questo metodo tiene conto anche delle richieste AJAX.
     *
     * @param string $redirect_url
     * @return JsonResponse|RedirectResponse
     */
    public function redirectResponse($redirect_url)
    {
        $redirect_url = absolute_url($redirect_url);

        if ($this->container->getService('request')->isAjax()) {
            return new JsonResponse(
                array(
                    'redirect' => $redirect_url,
                )
            );
        } else {
            return new RedirectResponse($redirect_url);
        }
    }

    /**
     * Restituisce la localizzazione.
     *
     * @return \Application\Core\Localization
     */
    public function getLocalization()
    {
        return $this->container->getService('language');
    }

    /**
     * Restituisce il modello della lingua.
     *
     * @return \Application\Core\Model\LanguageSite
     */
    public function getLanguage()
    {
        return $this->container->getService('language_site');
    }

    /**
     * Restituisce l'id della lingua
     *
     * @return int
     */
    public function getLanguageId()
    {
        return $this->getLanguage()->language_id;
    }

    /**
     * Resituisce il modello del sito corrente.
     *
     * @return \Application\Core\Model\Site
     */
    public function getSite()
    {
        return $this->container->getService('site');
    }

    /**
     * @return Firewall
     */
    public function getFirewall()
    {
        return $this->container->getService('firewall');
    }

    /**
     * Restituisce l’utente.
     *
     * @return \Application\Core\User
     */
    public function getUser()
    {
        return $this->getFirewall()->getCurrentUser();
    }

    /**
     * Restituisce l'id dell'utente
     */
    public function getUserId()
    {
        return $this->getUser()->getAccount()->id;
    }

    /**
     * Verifica se l'utente ha un permesso specifico
     *
     * @param $key
     * @return bool
     */
    public function userHasPermit($key)
    {
        return $this->getUser()->hasPermit($key);
    }

    /**
     * Restituisce il modulo legato al sito.
     *
     * @param boolean $throw_exception
     * @return \Application\Core\Model\ModuleSite
     */
    public function getModuleSite($throw_exception = true)
    {
        if ($this->module_site === null) {
            $this->module_site = $this->container->getParameter('site_module', null, $throw_exception);
        }

        return $this->module_site;
    }

    /**
     * @return Translator
     */
    public function getTranslator()
    {
        return $this->container->getService('translator');
    }
}
