<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core;

use Application\Core\Entity\AccountInterface;
use Application\Core\Model\Account;

/**
 * Rappresenta l’utente corrente.
 */
class User
{
    /**
     * Modello Account.
     *
     * @var AccountInterface
     */
    private $data;

    /**
     * @param $user_id
     */
    public function setUserId($user_id = null)
    {
        if ($user_id === null || intval($user_id) <= 0) {
            $user_id = Account::ANONYMOUS;
        }

        $this->setData($user_id);
    }

    /**
     * @return int
     */
    public function getUserId()
    {
        if ($this->data) {
            return $this->data->getId();
        }

        return Account::ANONYMOUS;
    }

    /**
     * Cerca l’utente corrente e crea l’istanza con i dati in database.
     *
     * @param $account_or_id
     */
    private function setData($account_or_id)
    {
        if ($account_or_id instanceof AccountInterface) {
            $this->data = $account_or_id;
        } else {
            $this->data = Account::find($account_or_id);

            if ($this->data === null || $this->data->isActive() === false) {
                $this->data = Account::find(Account::ANONYMOUS);
            }
        }
    }

    /**
     * Restituisce il modello dell’account legato all’utente corrente.
     *
     * @return AccountInterface
     */
    public function getAccount()
    {
        return $this->data;
    }

    /**
     * Restituisce un attributo del modello account dell’utente corrente.
     *
     * @param string $name
     * @return mixed
     */
    public function __get($name)
    {
        try {
            return $this->data->$name;
        } catch (\Exception $e) {
            return $name;
        }
    }

    /**
     * Esegue il login.
     *
     * @param string  $username_or_email
     * @param string  $password
     * @param array   $errors
     * @return boolean
     *
     * @deprecated
     */
    public function login(&$username_or_email, &$password, &$errors)
    {
        throw new \LogicException('This function is deprecated, use the Firewal::login()');
    }

    /**
     * Forza il login all’account passato per gestire l’accesso automatico dopo alcune operazioni particolari.
     *
     * Ovviamente, **DA USARE CON CAUTELA**!
     *
     * @param AccountInterface $account
     * @return $this
     */
    public function forceLogin(AccountInterface $account)
    {
        $this->setData($account);

        return $this;
    }

    /**
     * Esegue il logout dell’utente.
     *
     * @param boolean $continue Indica se dopo il logout lo script continua, oppure esce.
     * @return boolean
     */
    public function logout($continue = false)
    {
        if ($continue) {
            $this->data = Account::find(Account::ANONYMOUS);
        }
    }

    /**
     * Modifica i dati dell’utente corrente.
     *
     * @param array $data
     * @return boolean
     *
     * @deprecated
     */
    public function edit(array $data = array())
    {
        $this->data->update_attributes($data);

        if ($this->data->save()) {
            return true;
        }

        return $this->data->errors;
    }

    /**
     * Restituisce il nome dell'utente.
     *
     * @return string
     */
    public function name()
    {
        return $this->data->name();
    }

    /**
     * Restituisce true se l’utente è loggato.
     *
     * @return boolean
     */
    public function isLogged()
    {
        return ($this->data->getId() !== Account::ANONYMOUS);
    }

    /**
     * Restituisce true se l’utente è amministratore.
     *
     * @return boolean
     */
    public function isAdmin()
    {
        return $this->data->isAdmin();
    }

    /**
     * Restituisce true se l’utente è founder.
     *
     * @return boolean
     */
    public function isFounder()
    {
        return $this->data->isFounder();
    }

    /**
     * Restituisce true se l’utente ha il permesso indicato.
     *
     * @param string $key
     * @return boolean
     */
    public function hasPermit($key)
    {
        return $this->data->hasPermit($key);
    }

    /**
     * Verifica se l’utente appartiene ad un gruppo.
     *
     * È possibile specificare anche una lista di gruppi passando come argomento un array.
     * Il gruppo può essere sia istanza di Group che l'ID del gruppo.
     *
     * @param mixed $groups
     * @return self
     *
     * @deprecated
     */
    public function inGroup($groups)
    {
        return $this->data->inGroup($groups);
    }
}
