<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use Application\Core\Entity\SiteInterface;
use Application\Core\Model\Manager\LanguageSiteManagerInterface;
use Pongho\Core\Kernel;

/**
 * Modello per i siti.
 *
 * @property int    $id
 * @property string $name
 * @property string $domain
 * @property string $content_path
 * @property string $images_path
 * @property string $images_url
 * @property string $uploads_path
 * @property string $uploads_url
 * @property string $themes_path
 * @property string $themes_url
 * @property int    $default_language_id
 * @property string $settings
 */
class Site extends Settings implements SiteInterface
{
    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
	public static $table_name = 'sites';

	/**
	 * Lista delle lingue installate nel sito.
	 *
	 * @var array
	 */
	private $languages;

	/**
	 * Lingua di base del sito.
	 *
	 * @var \Application\Core\Model\Language
	 */
	private $default_language;

	/**
	 * Lista delle lingue e dei moduli installati nel sito.
	 *
	 * @var \Application\Core\Model\LanguageModule[]
	 */
	private $languages_modules;

    /**
     * Callback `before_save`.
     *
     * @var array
     * @static
     */
    public static $before_save = array('prepareSettingsOnSave', 'trimPaths');

    /**
     * Validazioni 'validates_by_methods_of'.
     *
     * @var array
     * @static
     */
    public static $validates_by_methods_of = array(
        array('content_path', 'method' => 'validatePath', 'message' => 'invalid_content_path'),
        array('images_path', 'method' => 'validatePath', 'message' => 'invalid_images_path'),
        array('uploads_path', 'method' => 'validatePath', 'message' => 'invalid_uploads_path'),
        array('themes_path', 'method' => 'validatePath', 'message' => 'invalid_themes_path'),
    );

	/**
	 * Restituisce le lingue abilitate in questo sito.
	 *
	 * La lista restituita contiene modelli di `Application\Core\Model\LanguageSite`.
	 *
	 * @return \Application\Core\Model\LanguageSite[]
	 */
    public function getLanguages()
    {
//        /** @var LanguageSiteManagerInterface $language_site_manager */
//        $language_site_manager = Kernel::instance()->getContainer()->getService('language_site_manager');


        if ($this->languages === null) {
            $this->languages = LanguageSite::all(
                array(
//				'select'		=> '`from`.*, l.*, `from`.id AS language_site_id',
                    'select'     => 'l.*, `from`.id AS language_site_id, `from`.*',
                    'joins'      => 'LEFT JOIN ' . Language::tableName() . ' AS l ON l.id = `from`.language_id',
                    'conditions' => array(
                        '`from`.site_id = :site AND `from`.is_enabled = :enabled',
                        'site'    => $this->id,
                        'enabled' => true
                    ),
                )
            );
        }

        return $this->languages;
    }

	/**
	 * Restituisce la lingua di default.
	 *
	 * @return \Application\Core\Model\LanguageSite
	 */
	public function getDefaultLanguage()
	{
//        /** @var LanguageSiteManagerInterface $language_site_manager */
//        $language_site_manager = Kernel::instance()->getContainer()->getService('language_site_manager');
//
//        return $language_site_manager->findByLanguageAndSite($this->default_language_id, $this->id);

		if ( $this->default_language === null )
		{
			$this->default_language = LanguageSite::first(array(
				'joins'			=> 'JOIN ' . Language::tableName() . ' AS l ON l.id = `from`.language_id',
				'conditions'	=> array('`from`.language_id = :language AND `from`.site_id = :site', 'language' => $this->default_language_id, 'site' => $this->id),
			));
		}

		return $this->default_language;
	}

	/**
	 * Restituisce una lista di moduli e lingue installate nel sito.
	 *
	 * La lista restituita contiene modelli di `Application\Core\Model\LanguageModule`.
	 *
	 * @return \Application\Core\Model\LanguageModule[]
	 */
	public function getLanguagesModules()
	{
		if ( $this->languages_modules === null )
		{
			$this->languages_modules = LanguageModule::all(array(
				'select'		=> '`from`.*, ls.language_id, ls.site_id',
				'joins'			=> 'INNER JOIN ' . ModuleSite::tableName() . ' AS ms ON ms.module_id = `from`.module_id'
                                . ' INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id AND ls.site_id = ms.site_id',
				'conditions'	=> array('ls.site_id = :site AND ms.is_enabled = :enabled AND ls.is_enabled = :enabled', 'site' => $this->id, 'enabled' => true),
				'order'			=> 'position',
			));
		}

		return $this->languages_modules;
	}

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getDefaultLanguageId()
    {
        return $this->default_language_id;
    }

    /**
     * Restituisce il nome del sito.
     *
     * @return string
     * @deprecated
     */
	public function name()
	{
		return $this->getName();
	}

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Restituisce il link alla home page del sito.
     *
     * Esempi:
     * - `http://www.example.com`
     * - `http://www.example.com/site`
     *
     * @return string
     * @deprecated
     */
    public function permalink()
    {
        return $this->getPermalink();
    }

    /**
     * {@inheritdoc}
     */
    public function getPermalink()
    {
        return $this->getDomain() . '/';
    }

    /**
     * Restituisce il dominio del sito.
     *
     * Esempi:
     * - `http://www.example.com/`
     * - `http://www.example.com/site/`
     *
     * @return string
     * @deprecated
     */
    public function domain()
    {
        return $this->getDomain();
    }

    /**
     * {@inheritdoc}
     */
    public function getDomain()
    {
        $domain = $this->domain;

        if ($this->hasOption('domain')) {
            $domain = $this->getOption('domain');
        }

        return $domain;
    }

    /**
     * {@inheritdoc}
     */
    public function getContentPath($path = '')
    {
        $content_path = $this->content_path;

        if ($this->hasOption('content_path')) {
            $content_path = $this->getOption('content_path');
        }

        if ($path !== '') {
            $path = '/' . ltrim($path, '/');
        }

        if (substr($content_path, 0, strlen(ROOT_PATH)) === ROOT_PATH) {
            return realpath($content_path) . $path;
        }

        return realpath(ROOT_PATH . $content_path) . $path;
    }

    /**
     * Restituisce il percorso della cartella dei plugins.
     *
     * @return string
     * @deprecated Usare direttamente la costante PLUGINS_PATH
     */
    public function getPluginsPath()
    {
        return PLUGINS_PATH;
    }

    /**
     * {@inheritdoc}
     */
    public function getThemesPath($path = '')
    {
        $themes_path = $this->themes_path;

        if ($this->hasOption('themes_path')) {
            $themes_path = $this->getOption('themes_path');
        }

        if ($path !== '') {
            $path = '/' . ltrim($path, '/');
        }

        if (substr($this->themes_path, 0, strlen(ROOT_PATH)) === ROOT_PATH) {
            return realpath($themes_path) . $path;
        }

        return realpath(ROOT_PATH . $themes_path) . $path;
    }

    /**
     * {@inheritdoc}
     */
    public function getImagesPath($path = '')
    {
        $images_path = $this->images_path;

        if ($this->hasOption('images_path')) {
            $images_path = $this->getOption('images_path');
        }

        if ($path !== '') {
            $path = '/' . ltrim($path, '/');
        }

        if (substr($images_path, 0, strlen(ROOT_PATH)) === ROOT_PATH) {
            return realpath($images_path) . $path;
        }

        return realpath(ROOT_PATH . $images_path) . $path;
    }

    /**
     * {@inheritdoc}
     */
    public function getUploadsPath($path = '')
    {
        $uploads_path = $this->uploads_path;

        if ($this->hasOption('uploads_path')) {
            $uploads_path = $this->getOption('uploads_path');
        }

        if ($path !== '') {
            $path = '/' . ltrim($path, '/');
        }

        if (substr($uploads_path, 0, strlen(ROOT_PATH)) === ROOT_PATH) {
            return realpath($uploads_path) . $path;
        }

        return realpath(ROOT_PATH . $uploads_path) . $path;
    }

    /**
     * {@inheritdoc}
     */
    public function getImagesUrl($path = '')
    {
        $url = $this->images_url;

        if ($this->hasOption('images_url')) {
            $url = $this->getOption('images_url');
        }

        if ($path !== '') {
            $path = '/' . ltrim($path, '/');
        }

        return $url . $path;
    }

    /**
     * {@inheritdoc}
     */
    public function getUploadsUrl($path = '')
    {
        $url = $this->uploads_url;

        if ($this->hasOption('uploads_url')) {
            $url = $this->getOption('uploads_url');
        }

        if ($path !== '') {
            $path = '/' . ltrim($path, '/');
        }

        return $url . $path;
    }

    /**
     * {@inheritdoc}
     */
    public function getThemesUrl($path = '')
    {
        $url = $this->themes_url;

        if ($this->hasOption('themes_url')) {
            $url = $this->getOption('themes_url');
        }

        if ($path !== '') {
            $path = '/' . ltrim($path, '/');
        }

        return $url . $path;
    }

	/**
	 * {@inheritdoc}
	 */
	public function prepareSettingsOnCreate()
	{
		parent::prepareSettingsOnCreate();

		if ( file_exists(CONFIG_PATH . '/config_dev.php') )
		{
			$config = include CONFIG_PATH . '/config_dev.php';

			if ( is_array($config) )
			{
                foreach ($config as $key => $value) {
                    if (substr($key, 0, 5) === 'site_') {
                        $attribute = substr($key, 5);
                        $this->$attribute = $value;
                    }
                }

				$this->_settings = array_merge($this->_settings, $config);
			}
		}
	}

    /**
     * Rimuove lo slash `/` finale dal dominio e dai percorsi.
     */
    public function trimPaths()
    {
        $fields = array('domain', 'images_path', 'images_url', 'uploads_path', 'uploads_url', 'themes_path', 'themes_url');

        foreach ($fields as $field) {
            $this->$field = rtrim($this->$field, '/');
        }
    }

    /**
     * Verifica che il percorso esista nel file system.
     *
     * @param string $path
     * @return bool
     *
     * @ignore
     */
    public function validatePath($path)
    {
        if (file_exists($path)) {
            return true;
        }

        if (file_exists(ROOT_PATH . $path)) {
            return true;
        }

        return false;
    }

	/**
     * {@inheritdoc}
	 */
	public function getDefaultOptions()
	{
		return array(

			// Principale
			'site_name'					=> '',
			'site_domain'				=> '',
			'default_language'			=> 1,
			'subscription_mode'			=> 'email',
			'slogan'					=> 'Questo è un altro sito in Pongho',

			'recaptcha'					=> false,
			'recaptcha_private_key'		=> '6LeLbNISAAAAAJYL-Gcw6Psmx3kw1JO2xVK4iGW7',
			'recaptcha_public_key'		=> '6LeLbNISAAAAACpDE84OahSKXRNnEx8X9MWgjauv',

			// Azienda
			'company_name'				=> '',
			'company_type'				=> '',
			'company_ceo_name'			=> '',

			'company_address'			=> '',
			'company_postcode'			=> '',
			'company_city'				=> '',
			'company_city_link'			=> '',
			'company_province'			=> '',
			'company_province_plate'	=> '',
			'company_country'			=> '',

			'company_partita_iva'		=> '',
			'company_codice_fiscale'	=> '',
			'company_rea'				=> '',
			'company_camera'			=> '',
			'company_cap_soc'			=> '€ 0.000,00',
			'company_website'			=> '',

			'company_email'				=> '',
			'company_tel'				=> '',
			'company_tel2'				=> '',
			'company_tel3'				=> '',
			'company_fax'				=> '',
			'company_fax2'				=> '',
			'company_fax3'				=> '',

			'company_copyright'			=> '',

			// Google Maps
			'company_latitude'			=> '00.000000000000000',
			'company_longitude'			=> '00.000000000000000',

			// Social
			'fb'						=> false,
			'fb_comments'				=> false,
			'fb_appid'					=> '',
			'fb_appsecret'				=> '',
			'fb_admin_id'				=> '1346270717',
			'fb_og_type'				=> 'website',

			// SEO
			'allow_robots'				=> false,
			'google_analytics'			=> 'UA-XXXXX-XX',

			'gapi_email'				=> 'webagencymetaline@gmail.com',
			'gapi_password'				=> '',
			'gapi_profile_id'			=> '123456789',

			// Autore del sito
			'author_name'				=> 'Web Agency Meta Line S.r.l.',
			'author_email'				=> 'info@metaline.it',
			'author_website'			=> 'http://www.metaline.it/',

			// Percorsi
			'java_path'					=> '/usr/bin/java',
			'ruby_path'					=> null,
			'sass_path'					=> '/usr/bin/sass',

		);
	}
}
