<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\SettingsInterface;

/**
 * Modello base per moduli e siti che necessitano di impostazioni.
 *
 * @property string $settings
 */
abstract class Settings extends Base implements SettingsInterface
{
    /**
     * @var array
     */
    protected $_settings;

    /**
     * Callback 'after_construct'.
     *
     * @var array
     * @static
     */
    public static $after_construct = array('prepareSettingsOnCreate');

    /**
     * Callback 'before_save'.
     *
     * @var array
     * @static
     */
    public static $before_save = array('prepareSettingsOnSave');

    /**
     * Prepara le opzioni per la lettura.
     */
    public function prepareSettingsOnCreate()
    {
        $settings = unserialize($this->settings);
        $settings = is_array($settings) ? $settings : array();

        $this->_settings = array_merge($this->getDefaultOptions(), $settings);
    }

    /**
     * Prepara le opzioni per il salvataggio in database.
     */
    public function prepareSettingsOnSave()
    {
        $this->settings = serialize($this->_settings);
    }

    /**
     * Imposta le opzioni di default.
     *
     * @param array $options
     * @return $this
     */
    public function setDefaultOptions(array $options)
    {
        foreach ($options as $name => $value) {
            if (!array_key_exists($name, $this->_settings)) {
                $this->_settings[$name] = $value;
            }
        }

        return $this;
    }

    /**
     * Indica se l’opzione esiste.
     *
     * @param $name
     * @return bool
     */
    public function hasOption($name)
    {
        return array_key_exists($name, $this->_settings);
    }

    /**
     * Restituisce una opzione.
     *
     * @param string    $name
     * @param null $default
     * @return null
     */
    public function getOption($name, $default = null)
    {
        return array_key_exists($name, $this->_settings) ? $this->_settings[$name] : $default;
    }

    /**
     * Imposta una opzione.
     *
     * @param string $name
     * @param mixed  $value
     */
    public function setOption($name, $value)
    {
        $this->_settings[$name] = $value;
    }

    /**
     * Elimina una opzione.
     *
     * @param string $name
     */
    public function delOption($name)
    {
        unset($this->_settings[$name]);
    }

    /**
     * Restituisce tutte le opzioni.
     *
     * @return array
     */
    public function allOptions()
    {
        return $this->_settings;
    }

    /**
     * Restituisce le opzioni di default.
     *
     * @return array
     */
    protected function getDefaultOptions()
    {
        return array();
    }
}
