<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\PermitInterface;
use Pongho\Core\Kernel;

/**
 * Modello per i permessi.
 *
 * @property int    $id
 * @property string $key
 * @property bool   $is_enabled
 */
class Permit extends Base implements PermitInterface
{
    /**
     * Nome della tabella.
     *
     * @static
     * @var string
     */
    public static $table_name = 'permits';

    /**
     * Relazioni 'has_and_belongs_to_many'.
     *
     * @static
     * @var array
     */
    public static $has_and_belongs_to_many = array(
        array('roles')
    );

    /**
     * Restituisce il modello del permesso in base alla chiave.
     *
     * @param string $key
     * @return \Application\Core\Model\Permit
     * @static
     */
    public static function findByKey($key)
    {
        return static::first(
            array(
                'conditions' => array('`key` = ?', $key)
            )
        );
    }

    /**
     * Restituisce i modelli dei permessi in base ad un prefisso.
     *
     * @param string $prefix
     * @return \Application\Core\Model\Permit[]
     * @static
     */
    public static function findByPrefix($prefix)
    {
        return static::all(
            array(
                'conditions' => array('pongho_like(`key`, ?)', $prefix . '_%)')
            )
        );
    }

    /**
     * @todo
     */
    public static function findBySuffix($suffix)
    {
    }

    /**
     * Cerca un permesso o lo crea.
     *
     * Il parametro $is_enabled viene utilizzato solo per la creazione.
     *
     * @param string $key
     * @param bool   $is_enabled
     * @return Base|Permit
     */
    public static function findOrCreate($key, $is_enabled = true)
    {
        $key = (string) $key;
        $is_enabled = (bool) $is_enabled;

        $permit = Permit::findByKey($key);

        if (!$permit) {
            $permit = Permit::create(
                array(
                    'key'        => $key,
                    'is_enabled' => $is_enabled,
                )
            );
        }

        return $permit;
    }

    /**
     * Abilita una lista di permessi in base alla loro chiave.
     *
     * Nel caso il permesso non esista questo viene creato.
     *
     * @param array $keys
     * @static
     */
    public static function enableKeys(array $keys)
    {
        foreach ($keys as $key) {
            if (($permit = static::findByKey($key)) === null) {
                static::create(
                    array(
                        'key'        => $key,
                        'is_enabled' => true,
                    )
                );
            } else {
                $permit->is_enabled = true;
                $permit->save(true);
            }
        }
    }

    /**
     * Disabilita una lista di permessi in base alla loro chiave.
     *
     * Nel caso il permesso non esista questo viene creato.
     *
     * @param array $keys
     * @static
     */
    public static function disableKeys(array $keys)
    {
        foreach ($keys as $key) {
            if (($permit = static::findByKey($key)) !== null) {
                $permit->is_enabled = false;
                $permit->save(true);
            }
        }
    }

    /**
     * Indica se un permesso è abilitato o meno.
     *
     * @param string $key
     * @return bool
     * @static
     */
    public static function isEnabled($key)
    {
        $permit = static::findByKey($key);

        if ($permit === null) {
            return false;
        }

        return $permit->is_enabled;
    }

    /**
     * @deprecated
     */
    public function name()
    {
        return Kernel::instance()->getContainer()->getService('language')->get('permit_' . $this->key);
    }

    /**
     * {@inheritdoc}
     */
    public function delete($in_recursion = false)
    {
        if ($this->deleteRolesRelations() && parent::delete()) {
            return true;
        }

        return false;
    }

    /**
     * Elimina le associazioni con i ruoli.
     *
     * @return bool
     */
    public function deleteRolesRelations()
    {
        $permit_roles = PermitRole::all(
            array(
                'conditions' => array('permit_id = :permit', 'permit' => $this->id)
            )
        );

        foreach ($permit_roles as $permit_role) {
            if (!$permit_role->delete()) {
                return false;
            }
        }

        return true;
    }
}
