<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Entity\NotificationTemplateInterface;
use Application\Core\Entity\SiteInterface;
use Application\Core\Model\NotificationTemplate;
use Application\Core\Model\NotificationTemplateTranslation;
use Application\Core\Notification\NotificationCollection;
use Application\Core\Notification\NotificationInterface;

/**
 * Class NotificationTemplateManager
 */
class NotificationTemplateManager implements NotificationTemplateManagerInterface
{
    /**
     * @var NotificationTemplateInterface[]
     */
    protected $models = array();

    /**
     * @var SiteInterface
     */
    protected $site;

    /**
     * @var NotificationCollection
     */
    protected $collection;

    /**
     * @param SiteInterface          $site
     * @param NotificationCollection $collection
     */
    public function __construct(SiteInterface $site, NotificationCollection $collection)
    {
        $this->site = $site;
        $this->collection = $collection;
    }

    /**
     * {@inheritdoc}
     */
    public function findById($id, LanguageInterface $language = null)
    {
        $conditions = array();

        if ($language) {
            $conditions = array(
                'ntt.language_id = :language',
                'language' => $language->getId()
            );
        }

        $options = $this->getDefaultQueryOptions($conditions);
        $options = NotificationTemplate::addCondition($options, array('id = :id', 'id' => $id));

        // TODO: cache
        return NotificationTemplate::first($options);
    }

    /**
     * {@inheritdoc}
     */
    public function findByNotification($notification_name, LanguageInterface $language = null)
    {
        if (!array_key_exists($notification_name, $this->models)) {
            $notification = $this->collection->get($notification_name);

            $this->models[$notification_name] = $notification->getDefaultTemplate();
        }

        return $this->models[$notification_name];
    }

    /**
     * {@inheritdoc}
     */
    public function findAll(array $conditions = array(), $order_by = null, $limit = null, $offset = null)
    {
        $models = array();

        /** @var NotificationInterface $notification */
        foreach ($this->collection as $notification) {
            $models[$notification->getName()] = new NotificationTemplate(
                array(
                    'notification' => $notification->getName(),
                )
            );
        }

        $options = $this->getDefaultQueryOptions($conditions, $order_by, $limit, $offset);

        foreach ($this->query($options) as $model) {
            $models[$model->getNotificationName()] = $model;
        }

        return $this->getIdentityMap($models);
    }

    /**
     * {@inheritdoc}
     */
    public function countAll(array $conditions = array())
    {
        $options = $this->getDefaultQueryOptions($conditions);

        return NotificationTemplate::count($options);
    }

    /**
     * @param array $options
     * @return NotificationTemplateInterface[]
     */
    protected function query($options)
    {
        return NotificationTemplate::all($options);
    }

    /**
     * @param NotificationTemplateInterface[] $records
     * @return NotificationTemplateInterface[]
     */
    protected function getIdentityMap(array $records)
    {
        $models = array();

        foreach ($records as $model) {
            if (!array_key_exists($model->getNotificationName(), $this->models)) {
                $this->models[$model->getNotificationName()] = $model;
                $models[] = $model;
            } else {
                $models[] = $this->models[$model->getNotificationName()];
            }
        }

        return $models;
    }

    /**
     * @param array  $conditions
     * @param string $order_by
     * @param int    $limit
     * @param int    $offset
     * @return array
     */
    protected function getDefaultQueryOptions(array $conditions = array(), $order_by = null, $limit = null, $offset = null)
    {
        $options = array(
            'select' => 'ntt.*, `from`.*',
            'joins'  => 'LEFT JOIN ' . NotificationTemplateTranslation::tableName() . ' AS ntt ON ntt.template_id = `from`.id',
        );

        $options = NotificationTemplate::addCondition($options, $conditions);
        $options = NotificationTemplate::addCondition(
            $options,
            array('`from`.site_id = :site', 'site' => $this->site->getId())
        );

        if ($order_by) {
            $options['order'] = $order_by;
        }

        if ($limit) {
            $options['limit'] = $limit;
        }

        if ($offset) {
            $options['offset'] = $offset;
        }

        return $options;
    }
}
