<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\ModuleSiteInterface;
use Application\Core\Model\ModuleSite;

/**
 * Class ModuleSiteManager
 */
class ModuleSiteManager extends AbstractCoreManager implements ModuleSiteManagerInterface
{
    /**
     * @var bool
     */
    protected $prepared = false;

    /**
     * @var string
     */
    static $model = 'ModuleSite';

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findById($module_site_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $module_site_id);

        if (!$model) {
            throw new \InvalidArgumentException(sprintf('The module site "%s" is undefined', $module_site_id));
        }

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function(ModuleSiteInterface $model) use ($site_id) {
                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByModule($module_id)
    {
        return array_filter(
            $this->findAll(),
            function(ModuleSiteInterface $model) use ($module_id) {
                if ($model->getModuleId() !== $module_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findBySiteAndModule($site_id, $module_id)
    {
        $model = array_filter(
            $this->findAll(),
            function(ModuleSiteInterface $model) use ($site_id, $module_id) {
                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                if ($model->getModuleId() !== $module_id) {
                    return false;
                }

                return true;
            }
        );

        if (empty($model)) {
            return null;
        }

        return reset($model);
    }

    /**
     * @param ModuleSiteInterface|ModuleSite $module_site
     * {@inheritdoc}
     */
    public function save(ModuleSiteInterface $module_site)
    {
        if (!$module_site->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if ($this->prepared) {
            return;
        }

        /** @var ModuleSiteInterface $model */
        foreach (ModuleSite::all() as $model) {
            $this->getCache()->set(static::$model, $model->getId(), $model);
        }

        $this->prepared = true;
    }
}
