<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\ModuleInterface;
use Application\Core\Model\Module;

/**
 * Class ModuleManager
 */
class ModuleManager extends AbstractCoreManager implements ModuleManagerInterface
{
    /**
     * @var bool
     */
    protected $prepared = false;

    /**
     * @var string
     */
    static $model = 'Module';

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function(ModuleInterface $model) use ($site_id) {
                $module_site = $this->registry->getModuleSiteManager()->findBySiteAndModule($site_id, $model->getId());

                if ($module_site === null) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByController($controller)
    {
        return array_filter(
            $this->findAll(),
            function(ModuleInterface $model) use ($controller) {
                if ($model->getController() !== $controller) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findById($module_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $module_id);

        if (!$model) {
            throw new \InvalidArgumentException(sprintf('The module "%s" is undefined', $module_id));
        }

        return $model;
    }

    /**
     * @param ModuleInterface|Module $module
     * {@inheritdoc}
     */
    public function save(ModuleInterface $module)
    {
        if (!$module->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if ($this->prepared) {
            return;
        }

        /** @var Module $model */
        foreach (Module::all() as $model) {
            $this->getCache()->set(static::$model, $model->getId(), $model);
        }

        $this->prepared = true;
    }
}
