<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\LanguageSiteInterface;
use Application\Core\Model\LanguageSite;

/**
 * Class LanguageSiteManager
 */
class LanguageSiteManager extends AbstractCoreManager implements LanguageSiteManagerInterface
{
    static $model = 'LanguageSite';

    /**
     * @var bool
     */
    protected $prepared = false;

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findById($language_site_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $language_site_id);

        if ($model === null) {
            throw new \InvalidArgumentException(sprintf('The language site "%s" is undefined', $language_site_id));
        }

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function findByLanguageAndSite($language_id, $site_id)
    {
        $models = array_filter(
            $this->findAll(),
            function (LanguageSite $model) use ($language_id, $site_id) {
                if ($model->getLanguageId() !== $language_id) {
                    return false;
                }

                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );

        if (empty($models)) {
            throw new \InvalidArgumentException(
                sprintf(
                    'The language site with language "%s" and site "%s" is undefined',
                    $language_id,
                    $site_id
                )
            );
        }

        return reset($models);
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function (LanguageSite $model) use ($site_id) {
                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * @param LanguageSiteInterface|LanguageSite $language_site
     * {@inheritdoc}
     */
    public function save(LanguageSiteInterface $language_site)
    {
        if (!$language_site->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function create($language_id, $site_id, $path, $is_enabled = true)
    {
        $this->prepare();

        $attributes = array(
            'language_id' => $language_id,
            'site_id'     => $site_id,
            'path'        => $path,
            'is_enabled'  => $is_enabled,
        );

        /** @var LanguageSiteInterface $language_site */
        if ($language_site = LanguageSite::create($attributes)) {
            $this->getCache()->set(static::$model, $language_site->getId(), $language_site);
        }

        return $language_site;
    }

    /**
     * {@inheritdoc}
     */
    public function enable(LanguageSiteInterface $language_site)
    {
        $language_site->enable();
        $this->save($language_site);
    }

    /**
     * {@inheritdoc}
     */
    public function disable(LanguageSiteInterface $language_site)
    {
        $language_site->disable();
        $this->save($language_site);
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if ($this->prepared) {
            return;
        }

        // fixme: rimuovere le opzioni di default, deve restituire modelli puliti, ma il problema è che altre classi usano LanguageSite come se fosse Language

        /** @var LanguageSite $language_site */
        foreach (LanguageSite::all(LanguageSite::getDefaultModelOptions()) as $language_site) {
            $this->getCache()->set(static::$model, $language_site->getId(), $language_site);
        }

        $this->prepared = true;
    }
}
