<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\LanguageInterface;
use Application\Core\Model\Language;
use Application\Core\Model\LanguageModule;

/**
 * Class LanguageManager
 */
class LanguageManager extends AbstractCoreManager implements LanguageManagerInterface
{
    /**
     * @var string
     */
    static $model = 'Language';

    /**
     * @var bool
     */
    protected $prepared = false;

    /**
     * {@inheritdoc}
     */
    public function findById($language_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $language_id);

        if (!$model) {
            throw new \InvalidArgumentException(sprintf('The language "%s" is undefined', $language_id));
        }

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByModuleAndSite($module_id, $site_id)
    {
        return array_filter(
            $this->findAll(),
            function(LanguageInterface $model) use ($site_id, $module_id) {
                $language_site = $model->getLanguageSite($site_id);

                if ($language_site === null) {
                    return false;
                }

                if ($language_site->getLanguageModule($module_id) === null) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     *
     * @todo: rivedere perché molto complessa
     */
    public function findAllByModuleSite($module_site_id)
    {
        $module_site = $this->registry->getModuleSiteManager()->findById($module_site_id);
        $language_modules = $module_site->getLanguageModules();

        return array_filter(
            $this->findAll(),
            function(LanguageInterface $model) use ($language_modules) {
                /** @var LanguageModule $language_module */
                foreach ($language_modules as $language_module) {
                    if ($language_module->getLanguage()->getId() !== $model->getId()) {
                        return false;
                    }
                }

                return true;
            }
        );
    }


    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function(LanguageInterface $model) use ($site_id) {
                try {
                    if ($model->getLanguageSite($site_id) === null) {
                        return false;
                    }
                } catch (\Exception $e) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * @param LanguageInterface|Language $language
     * {@inheritdoc}
     */
    public function save(LanguageInterface $language)
    {
        if (!$language->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * @return mixed
     */
    public function findFirst()
    {
        return reset($this->findAll());
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if ($this->prepared) {
            return;
        }

//        $options = array(
//            'select' => 'ls.id AS language_site_id, ls.site_id, ls.path AS ls_path, ls.is_enabled AS ls_enabled,'
//                     . ' lm.module_id, lm.position, lm.path AS lm_path, lm.is_enabled AS lm_enabled,'
//                     . ' `from`.*',
//            'joins'  => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.language_id = `from`.id'
//                     . ' LEFT JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.language_site_id = ls.id',
//        );
//
//        /** @var LanguageInterface $language */
//        foreach (Language::all($options) as $language) {
//            $this->languages[$language->getId()] = $language;
//        }

        /** @var Language $model */
        foreach (Language::all() as $model) {
            $this->getCache()->set(static::$model, $model->getId(), $model);
        }

        $this->prepared = true;
    }
}
